
/**
 * Author: hexuan
 * Date: 2024-12-24
 * illustrate: 2d 相机控制
 */

import { _decorator, Component, EventHandler, Camera, math, view, Size, Node, EventMouse, EventTouch, UITransform, Layers, Vec2, Vec3, v2, v3 } from 'cc';

const { ccclass, property } = _decorator;

@ccclass('Base2dCameraController')
export class Base2dCameraController extends Component {
    //#region property
    @property
    active: boolean = true;

    @property
    private _camera: Camera = null!;
    @property({
        type: Camera,
        visible: function () {
            //@ts-ignore
            return this.active;
        },
    })
    get camera() {
        return this._camera;
    }
    set camera(value: Camera) {
        this._camera = value;
        this._camera.projection = Camera.ProjectionType.ORTHO;
        this._camera.orthoHeight = this._visibleSize.height / 2;
        this._camera.near = 1;
        this._camera.far = 0;
        this._camera.visibility = Layers.Enum.MAP_2D;
    }
    @property({
        type: Node,
        visible: function () {
            //@ts-ignore
            return this.active;
        },
    })
    protected eventTarget: Node = null!;

    @property({
        tooltip: '相机移动边界，一般与地图大小一致',
        type: Size,
        visible: function () {
            //@ts-ignore
            let bool = this.active;
            if (bool) {
                //@ts-ignore
                bool = !this.free;
            }
            return bool;
        },
    })
    protected mapSize: Size = new Size(0);
    @property({
        type: EventHandler,
        visible: function () {
            //@ts-ignore
            return this.active;
        },
    })
    protected updatePositonZoomRatioEvents: EventHandler[] = [];

    @property({
        visible: function () {
            //@ts-ignore
            return this.active;
        },
    })
    protected free: boolean = false;

    //#endregion
    protected _visibleSize: Size = view.getVisibleSize();
    protected _orthoHeightTemp = this._visibleSize.height / 2;
    protected _positionTemp: Vec2 = new Vec2(0);
    protected _v2Temps: Vec2[] = [v2(0), v2(0), v2(0), v2(0)];
    protected _v3Temps: Vec3[] = [v3(0)];
    protected _needUpdate = false;
    protected _epsilon = 1; // 精度
    protected _moveCameraCompleteCallBack?: () => void;

    protected onEnable(): void {
        this.setTargetEvent(this.active);
    }
    protected onDisable(): void {
        this.setTargetEvent(false);
    }

    protected lateUpdate(dt: number): void {
        if (this._needUpdate) {
            const ratio = dt * 10;
            const x = math.lerp(this.camera.node.position.x, this._positionTemp.x, ratio);
            const y = math.lerp(this.camera.node.position.y, this._positionTemp.y, ratio);
            this.camera.orthoHeight = math.lerp(this.camera.orthoHeight, this._orthoHeightTemp, ratio);
            this.camera.node.setPosition(x, y);

            let subX = this._positionTemp.x - x;
            let subY = this._positionTemp.y - y;
            let length = Math.sqrt(subX * subX + subY * subY);
            if (length < this._epsilon && Math.abs(this._orthoHeightTemp - this.camera.orthoHeight) < this._epsilon) {
                this._needUpdate = false;
                this._moveCameraCompleteCallBack && this._moveCameraCompleteCallBack();
            }
            this.updatePositonZoomRatioEvents.length && Component.EventHandler.emitEvents(this.updatePositonZoomRatioEvents, this);
        }
    }

    protected setTargetEvent(_bool: boolean): void {
        const fun = _bool ? this.eventTarget.on : this.eventTarget.off;
        fun.call(this.eventTarget, Node.EventType.MOUSE_WHEEL, this._onMouseWheel, this);
        fun.call(this.eventTarget, Node.EventType.TOUCH_START, this._onTouchStart, this);
        fun.call(this.eventTarget, Node.EventType.TOUCH_MOVE, this._onTouchMove, this);
        fun.call(this.eventTarget, Node.EventType.TOUCH_END, this._onTouchEnd, this);
        fun.call(this.eventTarget, Node.EventType.TOUCH_CANCEL, this._onTouchEnd, this);
    }

    protected _onMouseWheel(e: EventMouse) {
        const scrollY = e.getScrollY();
        const zr = this._visibleSize.height / 2 / this._orthoHeightTemp;
        const orthoHeight = Math.max(0.000001, this._orthoHeightTemp - scrollY / 5 / zr);

        e.getLocation(this._v2Temps[0]);
        this._v3Temps[0].x = this._v2Temps[0].x;
        this._v3Temps[0].y = this._v2Temps[0].y;
        this.camera.screenToWorld(this._v3Temps[0], this._v3Temps[0]);
        this.camera.node.parent!.getComponent(UITransform)!.convertToNodeSpaceAR(this._v3Temps[0], this._v3Temps[0]);
        this._zoomWithPointer(this._v3Temps[0].x, this._v3Temps[0].y, orthoHeight);
        this._boundaryConstraint();
    }
    protected _onTouchStart(e: EventTouch) {
        this._positionTemp.x = this.camera.node.position.x;
        this._positionTemp.y = this.camera.node.position.y;
    }
    protected _onTouchMove(e: EventTouch) {
        const touches = e.getTouches();
        if (touches.length === 1) {
            // 单指移动
            const delta = e.getUIDelta(this._v2Temps[0]);
            const zoomRatio = this._visibleSize.height / 2 / this.camera.orthoHeight;
            this._positionTemp.x -= delta.x / zoomRatio;
            this._positionTemp.y -= delta.y / zoomRatio;
            this._boundaryConstraint();
        } else {
            // 双指缩放
            const location1 = touches[0].getLocation(this._v2Temps[0]);
            const location2 = touches[1].getLocation(this._v2Temps[1]);
            const delta1 = touches[0].getUIDelta(this._v2Temps[2]);
            const delta2 = touches[1].getUIDelta(this._v2Temps[3]);
        }
    }
    protected _onTouchEnd(e: EventTouch) {}

    private _zoomWithPointer(x: number, y: number, orthoHeight: number) {
        const zr = this.getCameraZoomRatio(this._orthoHeightTemp);
        const newzr = this.getCameraZoomRatio(orthoHeight);
        this._positionTemp.x = x - ((x - this._positionTemp.x) * zr) / newzr;
        this._positionTemp.y = y - ((y - this._positionTemp.y) * zr) / newzr;
        this._orthoHeightTemp = orthoHeight;
    }

    private _boundaryConstraint() {
        if (!this.free) {
            const zoomRatio = this.getCameraZoomRatio(this._orthoHeightTemp);
            const x = (this.mapSize.width - this._visibleSize.width / zoomRatio) / 2;
            const y = (this.mapSize.height - this._visibleSize.height / zoomRatio) / 2;
            this._positionTemp.x = math.clamp(this._positionTemp.x, -x, x);
            this._positionTemp.y = math.clamp(this._positionTemp.y, -y, y);
        }
        this._needUpdate = true;
        this._moveCameraCompleteCallBack = undefined;
    }


    public getCameraZoomRatio(orthoHeight: number = this.camera.orthoHeight) {
        return this._visibleSize.height / 2 / orthoHeight;
    }

    public setCameraPositonZoomRatio(x: number, y: number, zr: number, callback?: () => void) {
        this._orthoHeightTemp = this._visibleSize.height / 2 / zr;
        if (this.free) {
            this._positionTemp.x = x;
            this._positionTemp.y = y;
        } else {
            const limitX = (this.mapSize.width - this._visibleSize.width / zr) / 2;
            const limitY = (this.mapSize.height - this._visibleSize.height / zr) / 2;
            this._positionTemp.x = math.clamp(x, -limitX, limitX);
            this._positionTemp.y = math.clamp(y, -limitY, limitY);
        }
        this._needUpdate = true;
        this._moveCameraCompleteCallBack = callback;
    }
}
