/***
 *
 *   █████▒█    ██  ▄████▄   ██ ▄█▀       ██████╗ ██╗   ██╗ ██████╗
 * ▓██   ▒ ██  ▓██▒▒██▀ ▀█   ██▄█▒        ██╔══██╗██║   ██║██╔════╝
 * ▒████ ░▓██  ▒██░▒▓█    ▄ ▓███▄░        ██████╔╝██║   ██║██║  ███╗
 * ░▓█▒  ░▓▓█  ░██░▒▓▓▄ ▄██▒▓██ █▄        ██╔══██╗██║   ██║██║   ██║
 * ░▒█░   ▒▒█████▓ ▒ ▓███▀ ░▒██▒ █▄       ██████╔╝╚██████╔╝╚██████╔╝
 *  ▒ ░   ░▒▓▒ ▒ ▒ ░ ░▒ ▒  ░▒ ▒▒ ▓▒       ╚═════╝  ╚═════╝  ╚═════╝
 *  ░     ░░▒░ ░ ░   ░  ▒   ░ ░▒ ▒░
 *  ░ ░    ░░░ ░ ░ ░        ░ ░░ ░
 *           ░     ░ ░      ░  ░
 *
 *
 *	本代码作者：山水有相逢
 *	小号QQ:3179132247
 *	《json格式单词树》
 */

import { _decorator,warn } from 'cc';
const { ccclass, property } = _decorator;

@ccclass('JsonTree')
export class JsonTree {

    private static _instance: JsonTree = null;
    static get instance() {
        if (this._instance == null) this._instance = new JsonTree();
        return this._instance;
    }

    /**json数组转json单词树 */
    jsonArrToJsonTree(json: Object | any[]): Object {
        let outJsonTree: Object = {};
        let reg = /^[A-Za-z]+$/;
        for (const key in json) {
            let wordStr: string = json[key];
            if (reg.test(wordStr)) {
                this.addWordToTree(outJsonTree, wordStr.toLocaleLowerCase(), "1");
            } else {
                warn("单词不是纯字母：", wordStr);
            }
        }
        return outJsonTree;
    }

    /**
     * 添加单词进入json树
     * @param treeObj   json树
     * @param word      单词
     * @param nowIndex  当前查询序号
     */
    addWordToTree(treeObj: Object, word: string, value: any) {
        let lastObj: Object = treeObj;
        for (let index = 0; index < word.length; index++) {
            if (!(word[index] in lastObj)) lastObj[word[index]] = {};//如果没有指定键，创建并置空
            if (index == word.length - 1) { lastObj[word[index]]["_v"] = value; break; }//如果是单词结尾，填入值
            lastObj = lastObj[word[index]];//如果单词未结束，更新当前对象到下一级
        }
    }

    /**
     * 从json树移除单词
     * @param treeObj   json树
     * @param word      单词
     * @param nowIndex  当前查询序号
     */
    subWordToTree(treeObj: Object, word: string): boolean {
        let lastObj: Object = treeObj;
        for (let index = 0; index < word.length; index++) {
            if (!lastObj[word[index]]) {//如果没有目标键，移除失败
                return false;
            }
            lastObj = lastObj[word[index]];
        }
        delete lastObj["_v"];
        this.subWordPathToTree(treeObj, word);
        return true;
    }

    /**
     * 移除目标单词路径
     * @param treeObj   json树
     * @param word      单词
     */
    private subWordPathToTree(treeObj: Object, word: string) {
        for (let index = 0; index < word.length; index++) {
            let lastObj: Object = treeObj;
            for (let index = 0; index < word.length; index++) {
                if (!lastObj[word[index]]) break;
                if (JSON.stringify(lastObj[word[index]]) == "{}") { delete lastObj[word[index]]; break }
                lastObj = lastObj[word[index]];
            }
        }
    }

    /**
     * 查询单词是否在json单词树中
     * @param treeObj   单词json树 
     * @param word 要查询单词
     * @returns true：单词在树中，false：单词不在树中
     */
    findWordIsInTree(treeObj: Object, word: string): boolean {
        let lastObj: Object = treeObj;
        for (let index = 0; index < word.length; index++) {
            if (!lastObj[word[index]]) {//如果没有目标键，则未存入单词
                return false;
            }
            lastObj = lastObj[word[index]];
        }

        return !!lastObj["_v"];
    }
}
