import { assert, gfx, Material, renderer, rendering } from "cc";
import { CustomPipelineConst } from "./CustomPipelineConst";

export namespace CrtScanlineConst {
    export interface CrtScanline {
        enabled: boolean;
        material: Material | null;
        [name: string]: unknown;
    }

    export function fillRequiredCrtScanline(value: CrtScanline): void {
        if (value.material === undefined) {
            value.material = null;
        }
    }

    export interface CrtScanlinePassConfigs {
        enableCrtScanline: boolean;
    }

    /**
     * Custom CRT Scanline Pass Builder following Cocos Creator 3.8.6 Pipeline standards.
     */
    export class CustomCrtScanlinePassBuilder implements rendering.PipelinePassBuilder {
        private _runningTime: number = 0.0;
        getConfigOrder(): number {
            return 0;
        }
        getRenderOrder(): number {
            return 150;
        }

        configCamera(
            camera: Readonly<renderer.scene.Camera>,
            pplConfigs: Readonly<CustomPipelineConst.PipelineConfigs>,
            cameraConfigs: CustomPipelineConst.CameraConfigs & CrtScanlinePassConfigs): void {

            cameraConfigs.enableCrtScanline = cameraConfigs.settings.crtScanline.enabled
                && !!cameraConfigs.settings.crtScanline.material;

            if (cameraConfigs.enableCrtScanline) {
                cameraConfigs.enableStoreSceneDepth = true;
                ++cameraConfigs.remainingPasses;
            }
        }

        windowResize(
            ppl: rendering.BasicPipeline,
            pplConfigs: Readonly<CustomPipelineConst.PipelineConfigs>,
            cameraConfigs: Readonly<CustomPipelineConst.CameraConfigs & CrtScanlinePassConfigs>,
            window: renderer.RenderWindow): void {
            const id = window.renderWindowId;
            if (cameraConfigs.enableCrtScanline) {
                ppl.addRenderTarget(`CrtScanline${id}`,
                    cameraConfigs.radianceFormat,
                    cameraConfigs.width,
                    cameraConfigs.height);
            }
        }

        setup(
            ppl: rendering.BasicPipeline,
            pplConfigs: Readonly<CustomPipelineConst.PipelineConfigs>,
            cameraConfigs: CustomPipelineConst.CameraConfigs & CrtScanlinePassConfigs,
            camera: renderer.scene.Camera,
            context: CustomPipelineConst.PipelineContext,
            prevRenderPass?: rendering.BasicRenderPassBuilder)
            : rendering.BasicRenderPassBuilder | undefined {

            if (!cameraConfigs.enableCrtScanline) {
                return prevRenderPass;
            }
            --cameraConfigs.remainingPasses;

            if (cameraConfigs.remainingPasses == 0) {
                return this._addCrtScanlinePass(
                    ppl,
                    pplConfigs,
                    camera,
                    cameraConfigs.settings.crtScanline.material!,
                    cameraConfigs.renderWindowId,
                    cameraConfigs.width,
                    cameraConfigs.height,
                    context.colorName,
                    cameraConfigs.colorName
                );
            } else {
                const prefix = cameraConfigs.enableShadingScale
                    ? `ScaledRadiance`
                    : `Radiance`;
                const outputColorName = CustomPipelineConst.getPingPongRenderTarget(
                    context.colorName, prefix, cameraConfigs.renderWindowId);
                const inputColorName = context.colorName;
                context.colorName = outputColorName;
                return this._addCrtScanlinePass(
                    ppl,
                    pplConfigs,
                    camera,
                    cameraConfigs.settings.crtScanline.material!,
                    cameraConfigs.renderWindowId,
                    cameraConfigs.width,
                    cameraConfigs.height,
                    inputColorName,
                    outputColorName
                );
            }
        }

        private _addCrtScanlinePass(
            ppl: rendering.BasicPipeline,
            pplConfigs: Readonly<CustomPipelineConst.PipelineConfigs>,
            camera: renderer.scene.Camera,
            material: Material,
            id: number,
            width: number,
            height: number,
            inputTextureName: string,
            outputColorName: string) {

            assert(!!material);

            // 创建 Pass
            const pass = ppl.addRenderPass(width, height, "crt-scanline");

            // 设置输出目标 (LoadOp=CLEAR, StoreOp=STORE)
            pass.addRenderTarget(outputColorName, gfx.LoadOp.CLEAR, gfx.StoreOp.STORE, CustomPipelineConst.sClearColorTransparentBlack);

            // 设置视口
            pass.setViewport(new gfx.Viewport(0, 0, width, height));

            // 【关键修改】绑定输入纹理到 'mainTexture' (与 Shader 对应)
            pass.addTexture(inputTextureName, 'mainTexture');

            // 设置 Uniforms
            pass.setVec4('g_platform', pplConfigs.platform);

            this._runningTime += 0.02; // 简化时间计算，避免过快
            material.setProperty('u_time', this._runningTime);

            // 【关键修改】使用 addFullscreenQuad
            pass.addQueue(rendering.QueueHint.OPAQUE).addFullscreenQuad(material, 0);

            return pass;
        }
    }

    export function makeCrtScanline(): CrtScanline {
        return {
            enabled: false,
            material: null,
        };
    }
}