import { assert, rendering, geometry, renderer, gfx, Vec2, Vec4, sys, pipeline, Material, Color } from "cc";
import { makePipelineSettings, PipelineSettings } from "./CustomPipelineTypes";

export namespace CustomPipelineConst {
    export const defaultSettings = makePipelineSettings();
    export class PipelineConfigs {
        isWeb = false;
        isWebGL1 = false;
        isWebGPU = false;
        isMobile = false;
        isHDR = false;
        useFloatOutput = false;
        toneMappingType = 0; // 0: ACES, 1: None
        shadowEnabled = false;
        shadowMapFormat = gfx.Format.R32F;
        shadowMapSize = new Vec2(1, 1);
        usePlanarShadow = false;
        screenSpaceSignY = 1;
        supportDepthSample = false;
        mobileMaxSpotLightShadowMaps = 1;

        platform = new Vec4(0, 0, 0, 0);
    }

    export interface PipelineSettings2 extends PipelineSettings {
        _passes?: rendering.PipelinePassBuilder[];
    }

    export function forwardNeedClearColor(camera: renderer.scene.Camera): boolean {
        return !!(camera.clearFlag & (gfx.ClearFlagBit.COLOR | (gfx.ClearFlagBit.STENCIL << 1)));
    }

    export function getPingPongRenderTarget(prevName: string, prefix: string, id: number): string {
        if (prevName.startsWith(prefix)) {
            return `${prefix}${1 - Number(prevName.charAt(prefix.length))}_${id}`;
        } else {
            return `${prefix}0_${id}`;
        }
    }

    export function getCsmMainLightViewport(
        light: renderer.scene.DirectionalLight,
        w: number,
        h: number,
        level: number,
        vp: gfx.Viewport,
        screenSpaceSignY: number,
    ): void {
        if (light.shadowFixedArea || light.csmLevel === renderer.scene.CSMLevel.LEVEL_1) {
            vp.left = 0;
            vp.top = 0;
            vp.width = Math.trunc(w);
            vp.height = Math.trunc(h);
        } else {
            vp.left = Math.trunc(level % 2 * 0.5 * w);
            if (screenSpaceSignY > 0) {
                vp.top = Math.trunc((1 - Math.floor(level / 2)) * 0.5 * h);
            } else {
                vp.top = Math.trunc(Math.floor(level / 2) * 0.5 * h);
            }
            vp.width = Math.trunc(0.5 * w);
            vp.height = Math.trunc(0.5 * h);
        }
        vp.left = Math.max(0, vp.left);
        vp.top = Math.max(0, vp.top);
        vp.width = Math.max(1, vp.width);
        vp.height = Math.max(1, vp.height);
    }

    export function setupPipelineConfigs(
        ppl: rendering.BasicPipeline,
        configs: PipelineConfigs,
    ): void {
        const sampleFeature = gfx.FormatFeatureBit.SAMPLED_TEXTURE | gfx.FormatFeatureBit.LINEAR_FILTER;
        const device = ppl.device;
        // Platform
        configs.isWeb = !sys.isNative;
        configs.isWebGL1 = device.gfxAPI === gfx.API.WEBGL;
        configs.isWebGPU = device.gfxAPI === gfx.API.WEBGPU;
        configs.isMobile = sys.isMobile;

        // Rendering
        configs.isHDR = ppl.pipelineSceneData.isHDR; // Has tone mapping
        configs.useFloatOutput = ppl.getMacroBool('CC_USE_FLOAT_OUTPUT');
        configs.toneMappingType = ppl.pipelineSceneData.postSettings.toneMappingType;
        // Shadow
        const shadowInfo = ppl.pipelineSceneData.shadows;
        configs.shadowEnabled = shadowInfo.enabled;
        configs.shadowMapFormat = pipeline.supportsR32FloatTexture(ppl.device) ? gfx.Format.R32F : gfx.Format.RGBA8;
        configs.shadowMapSize.set(shadowInfo.size);
        configs.usePlanarShadow = shadowInfo.enabled && shadowInfo.type === renderer.scene.ShadowType.Planar;
        // Device
        configs.screenSpaceSignY = ppl.device.capabilities.screenSpaceSignY;
        configs.supportDepthSample = (ppl.device.getFormatFeatures(gfx.Format.DEPTH_STENCIL) & sampleFeature) === sampleFeature;
        // Constants
        const screenSpaceSignY = device.capabilities.screenSpaceSignY;
        configs.platform.x = configs.isMobile ? 1.0 : 0.0;
        configs.platform.w = (screenSpaceSignY * 0.5 + 0.5) << 1 | (device.capabilities.clipSpaceSignY * 0.5 + 0.5);
    }


    export class CameraConfigs {
        settings: PipelineSettings = defaultSettings;
        // Window
        isMainGameWindow = false;
        renderWindowId = 0;
        // Camera
        colorName = '';
        depthStencilName = '';
        // Pipeline
        enableFullPipeline = false;
        enableProfiler = false;
        remainingPasses = 0;
        // Shading Scale
        enableShadingScale = false;
        shadingScale = 1.0;
        nativeWidth = 1;
        nativeHeight = 1;
        width = 1; // Scaled width
        height = 1; // Scaled height
        // Radiance
        enableHDR = false;
        radianceFormat = gfx.Format.RGBA8;
        // Tone Mapping
        copyAndTonemapMaterial: Material | null = null;
        // Depth
        /** @en mutable */
        enableStoreSceneDepth = false; 
    }

    export const sClearColorTransparentBlack = new gfx.Color(0, 0, 0, 0);

    export function sortPipelinePassBuildersByConfigOrder(passBuilders: rendering.PipelinePassBuilder[]): void {
        passBuilders.sort((a, b) => {
            return a.getConfigOrder() - b.getConfigOrder();
        });
    }

    export function sortPipelinePassBuildersByRenderOrder(passBuilders: rendering.PipelinePassBuilder[]): void {
        passBuilders.sort((a, b) => {
            return a.getRenderOrder() - b.getRenderOrder();
        });
    }

    export function addCopyToScreenPass(
        ppl: rendering.BasicPipeline,
        pplConfigs: Readonly<PipelineConfigs>,
        cameraConfigs: CameraConfigs,
        input: string,
    ): rendering.BasicRenderPassBuilder {
        assert(!!cameraConfigs.copyAndTonemapMaterial);
        const pass = ppl.addRenderPass(
            cameraConfigs.nativeWidth,
            cameraConfigs.nativeHeight,
            'cc-tone-mapping');
        pass.addRenderTarget(
            cameraConfigs.colorName,
            gfx.LoadOp.CLEAR, gfx.StoreOp.STORE,
            sClearColorTransparentBlack);
        pass.addTexture(input, 'inputTexture');
        pass.setVec4('g_platform', pplConfigs.platform);
        pass.addQueue(rendering.QueueHint.OPAQUE)
            .addFullscreenQuad(cameraConfigs.copyAndTonemapMaterial, 1);
        return pass;
    }



    export interface PipelineContext {
        colorName: string;
        depthStencilName: string;
    }

}