import { assert, gfx, Material, renderer, rendering } from "cc";    
import { CustomPipelineConst } from "./CustomPipelineConst";
import { PipelineSettings } from "./CustomPipelineTypes";
 
export namespace RedTintConst {
    export interface RedTint {
        enabled: boolean;
        material: Material | null;
        intensity: number;
        [name: string]: unknown;
    } 



    export function makeRedTint(): RedTint {
        return {
            material: null,
            enabled: false,
            intensity: 0.5,
        }
    }

    export function fillRequiredRedTint(value: RedTint): void {
        if (value.material === undefined) {
            value.material = null;
        }
        if (value.intensity === undefined) {
            value.intensity = 0.5;
        }
    }

    export interface RedTintPassConfigs {
        enableRedTint: boolean;
    }

    export class RedTintPassBuilder implements rendering.PipelinePassBuilder {
        getConfigOrder(): number {
            return 0;
        }
        getRenderOrder(): number {
            return 650;
        }
        configCamera(
            camera: Readonly<renderer.scene.Camera>,
            pplConfigs: Readonly<CustomPipelineConst.PipelineConfigs>,
            cameraConfigs: CustomPipelineConst.CameraConfigs & RedTintPassConfigs): void {
            cameraConfigs.enableRedTint = cameraConfigs.settings.redTint.enabled
                && !!cameraConfigs.settings.redTint.material

            if (cameraConfigs.enableRedTint) {
                ++cameraConfigs.remainingPasses;
            }
        }

        setup(
            ppl: rendering.BasicPipeline,
            pplConfigs: Readonly<CustomPipelineConst.PipelineConfigs>,
            cameraConfigs: CustomPipelineConst.CameraConfigs & RedTintPassConfigs,
            camera: renderer.scene.Camera,
            context: CustomPipelineConst.PipelineContext,
            prevRenderPass?: rendering.BasicRenderPassBuilder)
            : rendering.BasicRenderPassBuilder | undefined {
            if (!cameraConfigs.enableRedTint) {
                return prevRenderPass;
            }
            --cameraConfigs.remainingPasses;

            const inputColorName = context.colorName;
            const outputColorName
                = cameraConfigs.remainingPasses === 0
                    ? cameraConfigs.colorName
                    : CustomPipelineConst.getPingPongRenderTarget(context.colorName, 'UiColor', cameraConfigs.renderWindowId);
            context.colorName = outputColorName;

            assert(!!cameraConfigs.settings.redTint.material);
            return this._addRedTintPass(ppl, pplConfigs, cameraConfigs,
                cameraConfigs.settings,
                cameraConfigs.settings.redTint.material,
                cameraConfigs.renderWindowId,
                cameraConfigs.width,
                cameraConfigs.height,
                inputColorName,
                outputColorName);
        }

        private _addRedTintPass(
            ppl: rendering.BasicPipeline,
            pplConfigs: Readonly<CustomPipelineConst.PipelineConfigs>,
            cameraConfigs: CustomPipelineConst.CameraConfigs & Readonly<RedTintPassConfigs>,
            settings: PipelineSettings,
            redTintMaterial: Material,
            id: number,
            width: number,
            height: number,
            inputTextureName: string,
            outputColorName: string) {

            if (!inputTextureName) {
                console.error('Red Tint: Invalid input texture name');
                const pass = ppl.addRenderPass(width, height, "red-tint");
                pass.addRenderTarget(outputColorName, gfx.LoadOp.CLEAR, gfx.StoreOp.STORE, CustomPipelineConst.sClearColorTransparentBlack);
                return pass;
            }

            console.log('Red Tint: inputTextureName =', inputTextureName, 'outputColorName =', outputColorName, 'width =', width, 'height =', height);

            const pass = ppl.addRenderPass(width, height, "red-tint");
            pass.addRenderTarget(outputColorName, gfx.LoadOp.CLEAR, gfx.StoreOp.STORE, CustomPipelineConst.sClearColorTransparentBlack);
            pass.addTexture(inputTextureName, 'inputTexture');
            pass.setVec4('g_platform', pplConfigs.platform);
            redTintMaterial.setProperty('u_intensity', settings.redTint.intensity);
            pass.addQueue(rendering.QueueHint.OPAQUE)
                .addFullscreenQuad(redTintMaterial, 0);

            return pass;
        }
    }
}