class HotUpdate {

    private _versionCompareFn: Function;
    set versionCompareFn(fn: Function) { this._versionCompareFn = fn; }
    private _isNeedUpdateFn: Function;
    set isNeedUpdateFn(fn: Function) { this._isNeedUpdateFn = fn; }
    private _updateFailFn: Function;
    set updateFailFn(fn: Function) { this._updateFailFn = fn; }
    private _updateSuccFn: Function;
    set updateSuccFn(fn: Function) { this._updateSuccFn = fn; }
    private _updateProgressFn: Function;
    set updateProgressFn(fn: Function) { this._updateProgressFn = fn; }

    private manifestUrl: cc.Asset = null;

    private strogeKey_curVersion: string = "curVersion";
    /**当前版本 */
    private _curVersion: string = "";

    /**存储路径 */
    private _storagePath: string = "";

    // 热更管理器
    private _assetMgr: jsb.AssetsManager = null;

    private isUpdating: boolean = false;

    /**初始化 */
    public init(manifest: cc.Asset, cb: (curV: string) => void): void {
        this.isUpdating = false;

        this.manifestUrl = manifest;
        this._curVersion = this.getCurVersion();
        this._storagePath = this.getRootPath();

        this._assetMgr = new jsb.AssetsManager("", this._storagePath, this.versionCompare.bind(this));
        this._assetMgr.setVerifyCallback(this.setVerifycb.bind(this));

        // 返回当前版本号
        cb && cb(this._curVersion);
    }

    private getCurVersion(): string {
        // 尝试从缓存中获取
        let curVersion = cc.sys.localStorage.getItem(this.strogeKey_curVersion);
        if (curVersion) return curVersion;
        let storagePath = this.getRootPath();
        storagePath = this.manifestUrl.nativeUrl;
        if (storagePath) {
            // 从本地缓存路径读取manifest文件
            cc.log(storagePath)
            let loadManifest = jsb.fileUtils.getStringFromFile(storagePath);
            let manifestObject = JSON.parse(loadManifest);
            curVersion = manifestObject.version;
        }
        return curVersion;
    }

    private getRootPath(): string {
        return (jsb.fileUtils ? jsb.fileUtils.getWritablePath() : "/") + "remote-asset";
    }

    /**比较版本号 */
    private versionCompare(v1: string, v2: string): number {
        this._versionCompareFn && this._versionCompareFn({ local: v1, server: v2 });
        console.log("================= 客户端版本：%s, 当前最新版本：%s ====================", v1, v2);
        let arr1 = v1.split(".");
        let arr2 = v2.split(".");
        for (let i = 0; i < arr1.length; ++i) {
            let num1 = parseInt(arr1[i]);
            let num2 = parseInt(arr2[i] || "0");
            if (num1 == num2) {
                continue;
            } else {
                return -1;
            }
        }
        if (arr1.length < arr2.length) {
            return -1;
        } else {
            return 0;
        }
    }

    private setVerifycb(assetFullPath: string, asset): boolean {
        // When asset is compressed, we don't need to check its md5, because zip file have been deleted.
        var compressed = asset.compressed;
        // Retrieve the correct md5 value.
        var expectedMD5 = asset.md5;
        // asset.path is relative path and path is absolute.
        var relativePath = asset.path;
        // The size of asset file, but this value could be absent.
        var size = asset.size;
        if (compressed) {
            console.log("Verification passed : " + relativePath);
            return true;
        }
        else {
            console.log("Verification passed : " + relativePath + ' (' + expectedMD5 + ')');
            return true;
        }
    }

    /**检测是否能够热更新 */
    public checkHotUpdate(): void {
        if (this.isUpdating) {
            return;
        }
        let url = this.manifestUrl.nativeUrl;
        console.log("原包版本信息url: ", url);
        if (this._assetMgr.getState() == jsb.AssetsManager.State.UNINITED) {
            if (cc.loader.md5Pipe) {
                url = cc.loader.md5Pipe.transformURL(url);
            }
            this._assetMgr.loadLocalManifest(url);
        }
        if (!this._assetMgr.getLocalManifest() || !this._assetMgr.getLocalManifest()) {
            console.log("加载本地manifest文件失败");
            return;
        }

        console.log("localManifest packageUrl：", this._assetMgr.getLocalManifest().getPackageUrl());

        this._assetMgr.setEventCallback(this.checkUpdateEvent.bind(this));
        this._assetMgr.checkUpdate();
        this.isUpdating = true;
    }

    private checkUpdateEvent(event: jsb.EventAssetsManager): void {
        console.log("checkUpdateEvent Code: %s", event.getEventCode());
        switch (event.getEventCode()) {
            case jsb.EventAssetsManager.ERROR_NO_LOCAL_MANIFEST:
                // this.label.string = "没有本地manifest文件，跳过热更.";
                this._isNeedUpdateFn && this._isNeedUpdateFn(false, "没有本地manifest文件，跳过热更.");
                break;
            case jsb.EventAssetsManager.ERROR_DOWNLOAD_MANIFEST:
            case jsb.EventAssetsManager.ERROR_PARSE_MANIFEST:
                // this.label.string = "下载远程manifest文件失败，跳过热更.";
                this._isNeedUpdateFn && this._isNeedUpdateFn(false, "下载远程manifest文件失败，跳过热更.");
                break;
            case jsb.EventAssetsManager.ALREADY_UP_TO_DATE:
                // this.label.string = "已经更新到远程最新版本.";
                this._isNeedUpdateFn && this._isNeedUpdateFn(false, "已经更新到远程最新版本.");
                break;
            case jsb.EventAssetsManager.NEW_VERSION_FOUND:
                // this.label.string = '发现新版本，请尝试热更';
                this._isNeedUpdateFn && this._isNeedUpdateFn(true, "发现新版本，请尝试热更");
                break;
            default:
                return;
        }
        this._assetMgr.setEventCallback(null);
        this.isUpdating = false;
    }

    /**热更新 */
    public hotUpdate(): void {
        if (this.isUpdating)
            return;
        if (!this._assetMgr)
            return;
        this._assetMgr.setEventCallback(this.hotUpdateEvent.bind(this));
        if (this._assetMgr.getState() == jsb.AssetsManager.State.UNINITED) {
            let url = this.manifestUrl.nativeUrl;
            if (cc.loader.md5Pipe) {
                url = cc.loader.md5Pipe.transformURL(url);
            }
            this._assetMgr.loadLocalManifest(url);
        }
        this._assetMgr.update();
        this.isUpdating = true;
    }

    private hotUpdateEvent(event: jsb.EventAssetsManager): void {
        console.log("hotUpdateEvent Code: [%s]  Msg: [%s]", event.getEventCode(), event.getMessage());
        switch (event.getEventCode()) {
            case jsb.EventAssetsManager.ERROR_NO_LOCAL_MANIFEST:
                // this.label.string = '没有本地manifest文件，跳过热更.';
                // failed = true;
                this.hotUpdateFail(event, "没有本地manifest文件，跳过热更.");
                break;
            case jsb.EventAssetsManager.UPDATE_PROGRESSION:
                console.log("当前下载文件数", event.getDownloadedFiles())
                console.log("总文件数", event.getTotalFiles())
                // var msg = event.getMessage();
                // if (msg) {
                //     this.label.string = '更新的文件：: ' + msg;
                // }
                this.hotUpdating(event, '更新的文件：: ' + event.getMessage());
                break;
            case jsb.EventAssetsManager.ERROR_DOWNLOAD_MANIFEST:
            case jsb.EventAssetsManager.ERROR_PARSE_MANIFEST:
                // this.label.string = '下载远程manifest文件失败，跳过热更.';
                // failed = true;
                this.hotUpdateFail(event, "下载远程manifest文件失败，跳过热更.");
                break;
            case jsb.EventAssetsManager.ALREADY_UP_TO_DATE:
                // this.label.string = '已经更新到远程最新版本.';
                // failed = true;
                this.hotUpdateFail(event, "已经更新到远程最新版本.");
                break;
            case jsb.EventAssetsManager.UPDATE_FINISHED:
                // this.label.string = '更新完成，即将重启游戏. ' + event.getMessage();
                // needRestart = true;
                this.hotUpdateSucc(event, '更新完成，即将重启游戏. ' + event.getMessage());
                break;
            case jsb.EventAssetsManager.UPDATE_FAILED:
                // this.label.string = '更新失败. ' + event.getMessage();
                // this.updating = false;
                // // this._canRetry = true;
                this.hotUpdateFail(event, '更新失败. ' + event.getMessage());
                break;
            case jsb.EventAssetsManager.ERROR_UPDATING:
                // this.label.string = 'Asset 更新错误: ' + event.getAssetId() + ', ' + event.getMessage();
                this.hotUpdateFail(event, 'Asset 更新错误: ' + event.getAssetId() + ', ' + event.getMessage());
                break;
            case jsb.EventAssetsManager.ERROR_DECOMPRESS:
                // this.label.string = event.getMessage();
                this.hotUpdateFail(event, event.getMessage());
                break;
            default:
                break;
        }
    }

    private hotUpdateFail(event: jsb.EventAssetsManager, msg: string): void {
        console.log("hotUpdateFail Msg: [%s]", msg);
        this.isUpdating = false;
        this._updateFailFn && this._updateFailFn(msg);
    }

    private hotUpdateSucc(event: jsb.EventAssetsManager, msg: string): void {
        this._assetMgr.setEventCallback(null);
        let searchPaths = jsb.fileUtils.getSearchPaths();
        let newPaths = this._assetMgr.getLocalManifest().getSearchPaths();
        Array.prototype.unshift(searchPaths, newPaths);
        cc.sys.localStorage.setItem("HotUpdateSearchPaths", JSON.stringify(searchPaths));
        jsb.fileUtils.setSearchPaths(searchPaths);

        let remoteVersion = this.getRemoteVersion();
        cc.sys.localStorage.setItem(this.strogeKey_curVersion, remoteVersion);

        this._updateSuccFn && this._updateSuccFn(remoteVersion);
    }

    private hotUpdating(event: jsb.EventAssetsManager, msg: string): void {
        let count_Download = event.getDownloadedFiles();
        let count_Total = event.getTotalFiles();

        this._updateProgressFn && this._updateProgressFn(count_Download / count_Total, msg);
    }

    private getRemoteVersion(): string {
        let storagePath = this.getRootPath();
        console.log("有下载的manifest文件", storagePath);
        let loadManifest = jsb.fileUtils.getStringFromFile(storagePath + "/project.manifest");
        let manifestObject = JSON.parse(loadManifest);
        return manifestObject.version;
    }

    /**自定义热更新 */
    public customerHotUpdate(): void {
        let curV = this.getCurVersion();
        this.reqHotUpdateInfo(curV, (info: { curV: string, tarV: string, tarPath: string, curPath: string }) => {
            cc.log(info);
            /* // 大版本更新 1.x.x -> 2.x.x 需要重新下载并安装apk
            let rootpath = this.getRootPath();
            // 清除之前缓存的版本号 和 版本路径
            cc.sys.localStorage.removeItem(this.strogeKey_curVersion);
            jsb.fileUtils.removeDirectory(rootpath);
            // 打开新版本应用下载页
            let packageUrl = "";
            cc.sys.openURL(packageUrl); */

            // 小版本更新
            let needUpdate = this.versionCompare(info.curV, info.tarV);
            if (needUpdate == 0) {
                console.log("小版本相同，不用更新");
                return;
            }
            this._modifyAppLoadUrlForManifestFile(info.tarPath, this.manifestUrl.nativeUrl);
        });
    }

    /**
    * 修改.manifest文件,如果有缓存则比较缓存的版本和应用包的版本，取较高者 热更，如果没有缓存则下载远程的版本文件 取版本号，这里统一修改地址为
    * 远程地址，不管 如何变化 都是从最新的地址下载版本文件。这里远程
    * @param {新的升级包地址} newAppHotUpdateUrl 
    * @param {本地project.manifest文件地址} localManifestPath 
    */
    private _modifyAppLoadUrlForManifestFile(newAppHotUpdateUrl, localManifestPath) {
        let isWritten = false;
        if (jsb.fileUtils.isFileExist(this.getRootPath() + '/project.manifest')) {
            let storagePath = this.getRootPath();
            console.log("有下载的manifest文件", storagePath);
            let loadManifest = jsb.fileUtils.getStringFromFile(storagePath + '/project.manifest');
            let manifestObject = JSON.parse(loadManifest);
            manifestObject.packageUrl = newAppHotUpdateUrl;
            manifestObject.remoteManifestUrl = newAppHotUpdateUrl + "/project.manifest";
            manifestObject.remoteVersionUrl = newAppHotUpdateUrl + "/version.manifest";
            let afterString = JSON.stringify(manifestObject);
            isWritten = jsb.fileUtils.writeStringToFile(afterString, storagePath + '/project.manifest');
        } else {
            /**
             * 执行到这里说明App之前没有进行过热更，所以不存在热更的plane文件夹。
             */

            /**
             * plane文件夹不存在的时候，我们就主动创建“plane”文件夹，并将打包时候的project.manifest文件中升级包地址修改后，存放到“plane”文件夹下面。
             */
            let initializedManifestPath = this.getRootPath();
            if (!jsb.fileUtils.isDirectoryExist(initializedManifestPath)) {
                jsb.fileUtils.createDirectory(initializedManifestPath);
            }
            //修改原始manifest文件
            let originManifestPath = localManifestPath;
            let originManifest = jsb.fileUtils.getStringFromFile(originManifestPath);
            let originManifestObject = JSON.parse(originManifest);
            originManifestObject.packageUrl = newAppHotUpdateUrl;
            originManifestObject.remoteManifestUrl = newAppHotUpdateUrl + '/project.manifest';
            originManifestObject.remoteVersionUrl = newAppHotUpdateUrl + '/version.manifest';

            let afterString = JSON.stringify(originManifestObject);
            isWritten = jsb.fileUtils.writeStringToFile(afterString, initializedManifestPath + '/project.manifest');
        }
        cc.log("Written Status : ", isWritten);
        if (isWritten) {
            this.checkHotUpdate();
        }

    }

    public reqHotUpdateInfo(curV: string, cb: Function): void {
        let xhr = cc.loader.getXMLHttpRequest();
        xhr.onreadystatechange = function () {
            if (xhr.readyState == 4 && xhr.status == 200) {
                let txt = xhr.responseText;
                console.log("get hotupdate info from server %s", txt);
                let data = JSON.parse(txt);
                cb && cb(data);
            }
        };

        xhr.withCredentials = false;
        xhr.open("GET", "http://192.168.3.94:3000//hotUpdate?curV=" + curV);
        xhr.send();
    }

    public reqLocalManifest(url: string, cb: Function): void {
        let xhr = cc.loader.getXMLHttpRequest();
        xhr.onreadystatechange = function () {
            if (xhr.readyState == 4 && xhr.status == 200) {
                let txt = xhr.responseText;
                console.log("get localManifest from server %s", txt);
                let data = JSON.parse(txt);
                cb && cb(data);
            }
        };
        xhr.withCredentials = false;
        xhr.open("GET", url, true);
        xhr.send();
    }
}

let hotInstance = new HotUpdate();
export default hotInstance;