import { ccenum, _decorator } from "cc";

export function polymorphism({
    types,
    displayName,
}: {
    types: Array<[new () => {
        constructor: Function;
    }, string]>;
    displayName?: string;
}): PropertyDecorator {
    return (target, propertyKey: PropertyKey) => {
        if (typeof propertyKey !== 'string') {
            throw new Error(`Only string named fields are supported.`);
        }

        const typePropertyKey = `${propertyKey}__type`;

        type This = {
            [x: typeof propertyKey]: {
                constructor: Function;
            };
        };

        const typePropertyDescriptor: PropertyDescriptor = {
            get(this: This) {
                const currentValue = this[propertyKey];
                const typeIndex = types.findIndex(([constructor]) => currentValue.constructor === constructor);
                if (typeIndex < 0) {
                    throw new Error(`${currentValue} is not a registered type.`);
                }
                return typeIndex;
            },

            set(this: This, value: number) {
                const [constructor] = types[value];
                if (this[propertyKey].constructor === constructor) {
                    return;
                }
                const object = new constructor();
                this[propertyKey] = object;
            }
        };

        const typeEnums = types.reduce((result, [constructor, name], index) => {
            result[name] = index;
            return result;
        }, {  } as Record<string, number>);
        ccenum(typeEnums);

        _decorator.property({
            displayName: displayName ?? `Type of ${propertyKey}`,
            type: typeEnums,
        })(target, typePropertyKey, typePropertyDescriptor);

        Object.defineProperty(target, typePropertyKey, typePropertyDescriptor);
    };
}
