/**
 * 可匹配房间列表信息
 * @param rooms - 单次请求的房间列表
 * @param count - 所有房间的总数
 * @param offset - 偏移量，作为下一次查询请求的入参
 * @param hasNext - 是否有下一页 0：无 1：有
 * @public
 */
export declare interface AvailableRoomsInfo {
    rooms: RoomInfo[];
    count: number;
    offset: string | number;
    hasNext: 0 | 1;
}

declare class Base {
    protected get state(): StateCode;
    protected get serverEventCode(): ServerEventCode;
    protected get accessToken(): string;
    protected get serviceToken(): string;
    get appId(): string;
    get clientId(): string;
    get clientSecret(): string;
    get roomId(): string;
    get groupId(): string;
    get openId(): string;
    get playerId(): string;
    protected constructor();
    protected setState(state: StateCode): void;
    protected setServerEvent(code: ServerEventCode, param?: string): void;
    protected setAccessToken(token: string): void;
    protected setServiceToken(token: string): void;
    protected setAppId(id: string): void;
    protected setOpenId(id: string): void;
    protected setClientId(id: string): void;
    protected setClientSecret(secret: string): void;
    protected setRoomId(id: string): void;
    protected setGroupId(id: string): void;
    protected setPlayerId(id: string): void;
    protected onStateChange(latter: StateCode, former: StateCode): void;
    protected onServerEvent(latter: ServerEvent, former: ServerEvent): void;
}

/**
 * 客户端类
 * @public
 */
export declare class Client extends Base {
    private _room;
    private _group;
    private _lastRoomId;
    private _isMatching;
    private _isCancelMatch;
    private _loginTimestamp;
    /**
     * 获取对应房间实例
     * @readonly
     */
    get room(): Room | null;
    /**
     * 获取对应队伍实例
     * @readonly
     */
    get group(): Group | null;
    /**
     * 上次断线的房间 ID
     * @readonly
     */
    get lastRoomId(): string;
    /**
     * 获取玩家登录时间戳
     * @readonly
     */
    get loginTimestamp(): number;
    /**
     * 创建客户端
     * @param config - 创建客户端参数
     */
    constructor(config: ClientConfig);
    /**
     * 初始化客户端
     * @remarks 必须先初始化客户端，才能创建/加入/匹配房间
     */
    init(): Promise<void>;
    /**
     * 创建房间
     * @remarks 创建成功也意味着加入了该房间
     * @param roomConfig - 房间信息参数
     * @param playerConfig - 玩家信息参数
     */
    createRoom(roomConfig: CreateRoomConfig, playerConfig?: PlayerConfig): Promise<Room>;
    /**
     * 创建队伍
     * @remarks 创建成功也意味着加入了该队伍
     * @param groupConfig - 队伍信息参数
     * @param playerConfig - 玩家信息参数
     */
    createGroup(this: Client, groupConfig: CreateGroupConfig, playerConfig?: PlayerConfig): Promise<Group>;
    /**
     * 加入房间
     * @param roomIdentity - 房间身份标识（房间Id或者房间Code）
     * @param playerConfig - 玩家信息参数
     */
    joinRoom(roomIdentity: string, playerConfig?: PlayerConfig): Promise<Room>;
    /**
     * 根据房间 ID 加入房间
     * @param roomId - 房间 ID
     * @param playerConfig - 玩家信息参数
     */
    joinRoomById(roomId: string, playerConfig?: PlayerConfig): Promise<Room>;
    /**
     * 根据房间 roomCode 加入房间
     * @param roomCode - 房间 code
     * @param playerConfig - 玩家信息参数
     */
    joinRoomByCode(roomCode: string, playerConfig?: PlayerConfig): Promise<Room>;
    /**
     * 重新加入房间
     */
    reconnectRoom(playerConfig?: PlayerConfig): Promise<Room>;
    /**
     * 根据队伍ID加入队伍
     * @param groupId - 队伍 ID
     * @param playerConfig - 玩家信息参数
     */
    joinGroup(groupId: string, playerConfig?: PlayerConfig): Promise<Group>;
    /**
     * 离开房间
     */
    leaveRoom(): Promise<Client>;
    /**
     * 解散房间
     * @remarks 房主才能解散房间
     */
    dismissRoom(): Promise<Client>;
    /**
     * 离开队伍
     */
    leaveGroup(): Promise<Client>;
    /**
     * 解散队伍
     * @remarks 队长才能解散队伍
     */
    dismissGroup(): Promise<Client>;
    /**
     * 删除队伍实例
     * @remarks 队员收到队伍解散通知后清空本地队伍信息
     */
    removeGroup(): void;
    /**
     * 获取可匹配房间列表
     */
    getAvailableRooms(getAvailableRoomsConfig: GetAvailableRoomsConfig): Promise<AvailableRoomsInfo>;
    /**
     * 房间匹配
     * @param matchRoomConfig -房间匹配参数
     * @param playerConfig - 玩家信息参数
     */
    matchRoom(matchRoomConfig: MatchRoomConfig, playerConfig?: PlayerConfig): Promise<Room>;
    /**
     * 在线匹配
     * @param matchPlayerConfig -在线匹配参数
     * @param playerConfig - 玩家信息参数
     * @remark 轮询时间间隔2s
     */
    matchPlayer(matchPlayerConfig: MatchPlayerConfig, playerConfig?: PlayerConfig): Promise<Room>;
    /**
     * 组队匹配
     * @param matchGroupConfig -组队匹配参数
     * @param playerConfig - 玩家信息参数
     * @remark 轮询时间间隔2s
     */
    matchGroup(matchGroupConfig: MatchGroupConfig, playerConfig?: PlayerConfig): Promise<Room>;
    /**
     * 取消匹配
     * @remark 组队匹配模式中，当前只有队长可以取消匹配
     *         该方法会修改_isCancelMatch状态 在matchPolling中请求真实的取消匹配接口并通过匹配接口返回取消结果
     */
    cancelMatch(): void;
    protected onStateChange(state: StateCode, former: StateCode): void;
    private checkInit;
    private checkCreateOrJoin;
    private checkGroupCreateOrJoin;
    private checkLeaveOrdismiss;
    private checkGroupLeaveOrdismiss;
    private checkCreateRoomConfig;
    private checkCreateGroupConfig;
    private checkJoinRoomConfig;
    private checkMatching;
    /**
     * 请求匹配接口
     */
    private requestMatch;
    /**
     * 请求取消匹配接口
     */
    private requestCancelMatch;
    /**
     * @remark 匹配轮询方法 包含取消匹配
     */
    private matchPolling;
}

/**
 * 客户端类构造方法参数
 * @param clientId - 客户端ID
 * @param clientSecret - 客户端密钥
 * @param appId - 应用ID
 * @param openId - 玩家ID
 * @public
 */
export declare interface ClientConfig {
    clientId: string;
    clientSecret: string;
    appId: string;
    openId: string;
}

/**
 * 创建队伍方法参数
 * @param maxPlayers - 队伍最大支持人数
 * @param groupName - 队伍名称
 * @param customGroupProperties - 队伍自定义属性
 * @param isLock - 是否禁止加入 0:不禁止 1:禁止 默认0
 * @param isPersistent - 是否持久化 0:不持久化 1:持久化 默认0
 * @public
 */
export declare interface CreateGroupConfig {
    maxPlayers: number;
    groupName?: string;
    customGroupProperties?: string;
    isLock?: number;
    isPersistent?: number;
}

/**
 * 创建房间方法参数
 * @param maxPlayers - 房间最大支持人数
 * @param isPrivate - 是否私有
 * @param roomType - 房间类型
 * @param roomName - 房间名称
 * @param matchParams - 房间匹配属性
 * @param customRoomProperties - 房间自定义属性
 * @public
 */
export declare interface CreateRoomConfig {
    maxPlayers: number;
    isPrivate?: number;
    roomType?: string;
    roomName?: string;
    matchParams?: Record<string, string>;
    customRoomProperties?: string;
}

/**
 * 错误码
 * @public
 */
export declare const enum ErrorCode {
    COMMON_OK = 0,
    COMMON_ERR = -1,
    COMMON_INVALID_TOKEN = 2,
    COMMON_REQUEST_PARAM_ERR = 1001,
    DATABASE_OPERATION_ERR = 1002,
    DCS_OPERATION_ERR = 1003,
    REPEAT_RUQUEST = 4003,
    TOKEN_AUTH_FAILED = 100003,
    SERVICE_NOT_ENABLED = 100004,
    PLAYER_INFO_MIASSING = 100005,
    PROJECT_NOT_EXIST = 100008,
    IP_AREA_ERR = 101001,
    PLAYER_AND_ROOM_MISMATCH = 101004,
    ROOM_INFO_NOT_EXIST = 101005,
    CANNOT_DESTORY_ROOM_IN_GAME = 101006,
    PLAYER_NOT_IN_CURRENT_ROOM = 101007,
    ENTERED_ROOM_OR_NOT_EXIST = 101008,
    PLAYERS_EXCEEDS_ROOM_MAX = 101009,
    PLAYER_INFO_NOT_EXIST = 101010,
    OWNER_AND_ROOM_MISMATCH = 101011,
    MAXPLAYER_TOO_LARGE = 101015,
    ROOM_NOT_BEGIN_FRAME_SYNC = 102003,
    PLAYER_NOT_IN_ROOM = 102005,
    ROOM_NOT_EXIST = 102008,
    PRASE_MESSAGE_FAILED = 102009,
    UNSUPPORTED_MESSAGE_TYPE = 102013,
    REQUEST_FRAME_NUMBER_OVERRUN = 102014,
    LOGIN_BUSY = 103001,
    LOGIN_AUTH_FAIL = 103002,
    CLIENT_TRAFFIC_CONTROL = 103003,
    NOT_LOGGED_IN = 103004,
    EXCEED_MAX_CONNECTIONSS = 103006,
    ROOM_MATCH_FAILED = 104001,
    ROOM_MATCHING = 104002,
    ROOM_MATCH_TIMEOUT = 104003,
    PLAYER_MATCH_FAILED = 306001,
    PLAYER_MATCHING = 306002,
    PLAYER_MATCH_TIMEOUT = 306003,
    PLAYER_MATCH_CANCEL_NO_PERMISSION = 306004,
    PLAYER_ID_NOT_CONSISTENT = 306005,
    PLAYER_MATCH_CANCEL_WHEN_SUCCESS = 306006,
    PLAYER_MATCH_GET_RULE_FAIL = 306007,
    PLAYER_MATCH_ROOM_IS_NULL = 306008,
    PLAYER_MATCH_CANCELED = 306009,
    SDK_NOT_IN_GROUP = 80001,
    SDK_GROUP_NAME_TOO_LONG = 80002,
    SDK_NO_PERMISSION_UPDATE_GROUP = 80003,
    SDK_IN_GROUP = 80004,
    SDK_UNINIT = 90001,
    SDK_NOT_IN_ROOM = 90002,
    SDK_IN_ROOM = 90003,
    SDK_NOT_ROOM_OWNER = 90004,
    SDK_NOT_IN_FRAME_SYNC = 90005,
    SDK_IN_FRAME_SYNC = 90006,
    SDK_INVALID_ROOM_IDENTITY = 90007,
    SDK_IN_MATCHING = 90008,
    SDK_ROOM_NAME_TOO_LONG = 10001
}

declare class EventEmitter<T extends (...args: any[]) => any> {
    handlers: Array<T>;
    on(handler: T): this;
    emit(...args: FunctionParam<T>): void;
    off(handler: T): void;
    clear(): void;
}

/**
 * 附加信息
 * @public
 */
export declare interface FrameExtInfo {
    seed: number;
}

/**
 * 帧数据信息
 * @public
 */
export declare interface FrameInfo extends FramePlayerInfo {
    data: string[];
    timestamp: number;
}

/**
 * 帧数据玩家信息
 * @public
 */
export declare interface FramePlayerInfo {
    playerId: string;
}

declare type FunctionParam<T> = T extends (...args: infer P) => any ? P : never;

/**
 * 获取可匹配房间列表参数
 * @param roomType - 房间类型
 * @param offset - 偏移量，使用房间的createTime作为每次请求的标记，第一次请求时为0
 * @param limit - 单次请求获取的房间数量，不选时服务端默认为20
 * @public
 */
export declare interface GetAvailableRoomsConfig {
    roomType?: string;
    offset?: number | string;
    limit?: number;
}

/**
 * 队伍类
 * @public
 */
export declare class Group extends Base {
    onJoin: {
        (this: any, cb: (serverEvent: ServerEvent) => void): EventEmitter<(serverEvent: ServerEvent) => void>;
        emit(serverEvent: ServerEvent): void;
        off(cb: (serverEvent: ServerEvent) => void): void;
        clear(): void;
    };
    onLeave: {
        (this: any, cb: (serverEvent: ServerEvent) => void): EventEmitter<(serverEvent: ServerEvent) => void>;
        emit(serverEvent: ServerEvent): void;
        off(cb: (serverEvent: ServerEvent) => void): void;
        clear(): void;
    };
    onDismiss: {
        (this: any, cb: (serverEvent: ServerEvent) => void): EventEmitter<(serverEvent: ServerEvent) => void>;
        emit(serverEvent: ServerEvent): void;
        off(cb: (serverEvent: ServerEvent) => void): void;
        clear(): void;
    };
    onUpdate: {
        (this: any, cb: (serverEvent: ServerEvent) => void): EventEmitter<(serverEvent: ServerEvent) => void>;
        emit(serverEvent: ServerEvent): void;
        off(cb: (serverEvent: ServerEvent) => void): void;
        clear(): void;
    };
    onMatchStart: {
        (this: any, cb: (serverEvent: ServerEvent) => void): EventEmitter<(serverEvent: ServerEvent) => void>;
        emit(serverEvent: ServerEvent): void;
        off(cb: (serverEvent: ServerEvent) => void): void;
        clear(): void;
    };
    onMatchComplete: {
        (this: any, cb: (serverEvent: ServerEvent, room?: Room | undefined) => void): EventEmitter<(serverEvent: ServerEvent, room?: Room | undefined) => void>;
        emit(serverEvent: ServerEvent, room?: Room | undefined): void;
        off(cb: (serverEvent: ServerEvent, room?: Room | undefined) => void): void;
        clear(): void;
    };
    private config;
    private _player;
    private _client;
    /**
     * 队伍 ID
     */
    get id(): string;
    /**
     * 队伍名称
     */
    get groupName(): string;
    /**
     * 队伍最大人数
     */
    get maxPlayers(): number;
    /**
     * 队长 ID
     */
    get ownerId(): string;
    /**
     * 队伍自定义属性
     */
    get customGroupProperties(): string;
    /**
     * 是否禁止加入 0:不禁止 1:禁止 默认0
     */
    get isLock(): number;
    /**
     * 是否持久化 0:不持久化 1:持久化 默认0
     */
    get isPersistent(): number;
    /**
     * 队伍玩家列表
     */
    get players(): PlayerInfo[];
    /**
     * 玩家自己
     */
    get player(): Player;
    /**
     * 队伍
     * @param config - 创建客户端参数
     */
    constructor(client: Client, config: GroupInfo, playerConfig?: PlayerConfig);
    /**
     * 队伍信息查询
     */
    query(): Promise<Group>;
    /**
     * 离开队伍
     */
    leave(): Promise<void>;
    /**
     * 解散队伍
     * @remarks 队长才能解散队伍
     */
    dismiss(): Promise<void>;
    /**
     * 更新队伍信息
     * @remarks 队长才能更新队伍信息
     * @param config - 更新队伍信息参数
     */
    updateGroup(config: UpdateGroupConfig): Promise<void>;
    private checkUpdatePermission;
    protected onServerEvent(serverEvent: ServerEvent): Promise<void>;
}

/**
 * 队伍信息
 * @param groupId - 队伍id
 * @param groupName - 队伍名称
 * @param maxPlayers - 最大玩家数
 * @param ownerId - 队长ID
 * @param customGroupProperties - 队伍自定义属性
 * @param isLock - 是否禁止加入 0:不禁止 1:禁止 默认0
 * @param isPersistent - 是否持久化 0:不持久化 1:持久化 默认0
 * @param players - 队伍内玩家列表
 * @public
 */
export declare interface GroupInfo {
    groupId: string;
    groupName: string;
    maxPlayers: number;
    ownerId: string;
    customGroupProperties: string;
    isLock: number;
    isPersistent: number;
    players: PlayerInfo[];
}

/**
 * 组队匹配参数
 * @param playerInfos - 带匹配规则的玩家信息列表
 * @param matchCode - 匹配规则编号
 */
declare interface MatchGroupConfig {
    playerInfos: MatchPlayerInfoParam[];
    matchCode: string;
}

/**
 * 在线匹配参数
 * @param playerInfo - 带匹配规则的玩家信息
 * @param matchCode - 匹配规则编号
 */
declare interface MatchPlayerConfig {
    playerInfo: MatchPlayerInfoParam;
    matchCode: string;
}

/**
 * 带匹配规则的玩家信息
 * @param matchParams - 自定义匹配参数，JSON对象格式
 */
declare interface MatchPlayerInfoParam {
    playerId: string;
    matchParams: Record<string, string>;
}

/**
 * 房间匹配参数
 * @param matchParams - 自定义匹配参数，JSON对象格式，最多支持5条匹配规则
 * @param maxPlayers - 房间最大支持人数
 * @param roomType - 房间类型
 * @param customRoomProperties - 自定义房间属性
 * @remarks maxPlayers，roomType，customRoomProperties用于找不到匹配房间时创建房间
 * @public
 */
export declare interface MatchRoomConfig {
    matchParams: Record<string, string>;
    maxPlayers: number;
    roomType?: string;
    customRoomProperties?: string;
}

/**
 * 玩家类
 * @public
 */
export declare class Player extends Base {
    private _customStatus?;
    private _customProperties?;
    constructor(customStatus?: number, customProperties?: string);
    get customPlayerStatus(): number | undefined;
    get customPlayerProperties(): string | undefined;
    /**
     * 修改玩家自定义状态
     */
    setCustomStatus(status: number): Promise<Player>;
}

/**
 * 玩家自定义参数
 * @param customPlayerStatus - 玩家自定义状态
 * @param customPlayerProperties - 玩家自定义属性
 * @public
 */
export declare interface PlayerConfig {
    customPlayerStatus?: number;
    customPlayerProperties?: string;
}

/**
 * 玩家信息
 * @param playerId - 玩家ID
 * @param status - 玩家状态 0：空闲；1：房间中；2：帧同步中；3：离线
 * @param customPlayerStatus - 自定义玩家状态
 * @param customPlayerProperties - 自定义玩家属性
 * @param teamId - 玩家teamId
 * @public
 */
export declare interface PlayerInfo {
    readonly playerId: string;
    readonly status?: number;
    readonly customPlayerStatus?: number;
    readonly customPlayerProperties?: string;
    readonly teamId?: string;
}

/**
 * 房间内消息码
 * @public
 */
export declare const enum Protocol {
    LOGIN = 0,
    LOGIN_ACK = 1,
    HEARTBEAT = 2,
    HEARTBEAT_ACK = 3,
    CLIENT_SEND_FRAMEDATA = 4,
    CLIENT_SEND_FRAMEDATA_ACK = 5,
    QUERY_FRAMEDATA = 6,
    QUERY_FRAMEDATA_ACK = 7,
    FRAMESYNC_STARTED = 8,
    FRAMESYNC_STOPED = 9,
    BROADCAST_FRAMEDATA = 10,
    QUERY_FRAMEDATA_RESULT = 17,
    JOIN_ROOM = 12,
    LEAVE_ROOM = 13,
    CONNECTED = 14,
    DISCONNECTED = 15,
    ROOM_DISMISS = 16
}

/**
 * 基于「线性同余」的伪随机数生成器
 * @public
 */
export declare class Random {
    private mask;
    private m;
    private a;
    private seed;
    constructor(seed: number);
    getNumber(): number;
}

/**
 * 房间类
 * @public
 */
export declare class Room extends Base {
    onJoin: {
        (this: any, cb: (player: FramePlayerInfo) => void): EventEmitter<(player: FramePlayerInfo) => void>;
        emit(player: FramePlayerInfo): void;
        off(cb: (player: FramePlayerInfo) => void): void;
        clear(): void;
    };
    onLeave: {
        (this: any, cb: (player: FramePlayerInfo) => void): EventEmitter<(player: FramePlayerInfo) => void>;
        emit(player: FramePlayerInfo): void;
        off(cb: (player: FramePlayerInfo) => void): void;
        clear(): void;
    };
    onDismiss: {
        (this: any, cb: () => void): EventEmitter<() => void>;
        emit(): void;
        off(cb: () => void): void;
        clear(): void;
    };
    onDisconnect: {
        (this: any, cb: (player: FramePlayerInfo) => void): EventEmitter<(player: FramePlayerInfo) => void>;
        emit(player: FramePlayerInfo): void;
        off(cb: (player: FramePlayerInfo) => void): void;
        clear(): void;
    };
    onStartFrameSync: {
        (this: any, cb: () => void): EventEmitter<() => void>;
        emit(): void;
        off(cb: () => void): void;
        clear(): void;
    };
    onStopFrameSync: {
        (this: any, cb: () => void): EventEmitter<() => void>;
        emit(): void;
        off(cb: () => void): void;
        clear(): void;
    };
    onRecvFrame: {
        (this: any, cb: (msg: ServerFrameMessage | ServerFrameMessage[]) => void): EventEmitter<(msg: ServerFrameMessage | ServerFrameMessage[]) => void>;
        emit(msg: ServerFrameMessage | ServerFrameMessage[]): void;
        off(cb: (msg: ServerFrameMessage | ServerFrameMessage[]) => void): void;
        clear(): void;
    };
    private connection;
    private config;
    private frameId;
    private endpoint;
    private heartbeatTimer;
    private _isSyncing;
    private _player;
    /**
     * 房间 ID
     */
    get id(): string;
    /**
     * 房间类型
     */
    get roomType(): string;
    /**
     * 房间名称
     */
    get roomName(): string;
    /**
     * 房间的短码
     */
    get roomCode(): string;
    /**
     * 房间自定义属性
     */
    get customRoomProperties(): string;
    /**
     * 房主 ID
     */
    get ownerId(): string;
    /**
     * 房间最大人数
     */
    get maxPlayers(): number;
    /**
     * 房间玩家列表
     */
    get players(): PlayerInfo[];
    /**
     * 路由信息
     */
    get router(): RouterInfo;
    /**
     * 0：公开房间，1：私有房间
     */
    get isPrivate(): number;
    /**
     * 创建时间
     */
    get createTime(): number;
    /**
     * 玩家自己
     */
    get player(): Player;
    /**
     * 房间是否处于帧同步
     */
    get isSyncing(): boolean;
    /**
     * 房间
     * @param config - 创建客户端参数
     */
    constructor(config: RoomInfo, playerConfig?: PlayerConfig);
    /**
     * websocket 建链
     * @param endpoint - 接入地址
     */
    connect(routerAddr: string, ticket: string): void;
    /**
     * 重连
     */
    reconnect(): Promise<void>;
    /**
     * 开始帧同步
     */
    startFrameSync(): Promise<void>;
    /**
     * 结束帧同步
     */
    stopFrameSync(): Promise<void>;
    /**
     * 发送帧数据
     * @param frameData - 帧数据
     */
    sendFrame(frameData: string | string[]): void;
    /**
     * 请求补帧
     * @param beginFrameId - 起始帧号
     * @param size - 请求帧号
     */
    requestFrame(beginFrameId: number, size: number): void;
    /**
     * 玩家房间信息查询
     */
    update(): Promise<Room>;
    /**
     * 离开房间
     */
    leave(): Promise<void>;
    /**
     * 解散房间
     * @remarks 房主才能解散房间
     */
    dismiss(): Promise<void>;
    /**
     * 踢人
     * @param playerId - 被踢出的玩家ID
     * @remarks 房主才能踢人
     * @remarks 房间在帧同步中，不能踢人
     */
    kickPlayer(playerId: string): Promise<void>;
    /**
     * 移除所有事件监听
     */
    removeAllListeners(): void;
    private onMessageCallback;
    private startHeartbeat;
    private doHeartbeat;
    private stopHeartbeat;
    private buildEndpoint;
    private checkIsInSync;
    private checkNotInSync;
}

/**
 * 房间信息
 * @public
 * @param appId - 游戏ID
 * @param roomId - 房间ID
 * @param roomType - 房间类型
 * @param roomCode - 房间的短码
 * @param roomName - 房间名称
 * @param roomStatus - 房间状态 0：空闲，1：帧同步中，2：待回收
 * @param customRoomProperties - 房间自定义属性
 * @param ownerId - 房主ID
 * @param maxPlayers - 房间最大支持人数
 * @param players - 房间内玩家
 * @param router - 路由信息
 * @param isPrivate - 是否私有
 * @param createTime - 创建时间
 */
export declare interface RoomInfo {
    appId: string;
    roomId: string;
    roomType: string;
    roomCode: string;
    roomName: string;
    roomStatus: number;
    customRoomProperties: string;
    ownerId: string;
    maxPlayers: number;
    players: PlayerInfo[];
    router: RouterInfo;
    isPrivate: number;
    createTime: number;
}

/**
 * 路由信息
 * @public
 */
export declare interface RouterInfo {
    routerId: number;
    routerType: number;
    routerAddr: string;
}

/**
 * 上报接口返回服务端事件
 * @param eventType - 事件类型 1：匹配开始；2：匹配成功；3：匹配失败；4：匹配取消；5：匹配超时；
 * 6：加入小队；7：离开小队；8：解散小队；9：更新小队；
 * @param eventParam - 事件相关信息
 * @public
 */
export declare interface ServerEvent {
    eventType: ServerEventCode;
    eventParam?: string;
}

declare const enum ServerEventCode {
    DEFAULT = 0,
    MATCH_START = 1,
    MATCH_SUCCESS = 2,
    MATCH_FAILED = 3,
    MATCH_CANCEL = 4,
    MATCH_TIMEOUT = 5,
    JOIN_GROUP = 6,
    LEAVE_GROUP = 7,
    DISMISS_GROUP = 8,
    UPDATE_GROUP = 9
}

/**
 * 服务端推送消息
 * @public
 */
export declare interface ServerFrameMessage {
    currentRoomFrameId: number;
    frameInfo: FrameInfo[];
    ext: FrameExtInfo;
}

declare const enum StateCode {
    UNINITIALIZED = 0,
    INITIALIZED = 1,
    INROOM = 2,
    SYNCING = 3,
    ENTER_ROOM = 4,
    LEAVE_ROOM = 5,
    ENTER_SYNCING = 6,
    EXIT_SYNCING = 7
}

/**
 * 更新队伍信息参数
 * @param groupName - 队伍名称
 * @param ownerId - 队长ID
 * @param customGroupProperties - 队伍自定义属性
 * @param isLock - 是否禁止加入 0:不禁止 1:禁止 默认0
 * @public
 */
export declare interface UpdateGroupConfig {
    groupName?: string;
    ownerId?: string;
    customGroupProperties?: string;
    isLock?: number;
}

export { }
export as namespace GOBE
