import { AssetManager, assetManager, director, Director, error, errorID, JsonAsset, Node, Prefab, resources, Scene, SceneAsset, Sprite, SpriteFrame, sys, warnID } from "cc";
import TxtMgr, { txt } from "./TxtMgr";
import I18LanguageHelper, { EBundleName } from "./I18LanguageHelper";
import { EI18AssetType, I18PreUrl } from "./I18PreUrl";
import { I18BtnUrl } from "./I18BtnUrl";
import { sp } from "cc";
import { Asset } from "cc";
import EventCenter from "../kernel/core/event/EventCenter";


export default class I18Mgr {
    static instance : I18Mgr = null;
    static getInstance(): I18Mgr{
        if( I18Mgr.instance == null ){
            I18Mgr.instance = new I18Mgr();
        }
        return I18Mgr.instance;
    }

    _currBundleName : EBundleName = EBundleName.NONE;
    _currBundle : AssetManager.Bundle = null;

    _mapSpriteFrame = new Map();
    _mapPrefabLoad = new Map();

    _mapRes = new Map<EI18AssetType, Map<string, any>>();

    constructor(){
        // const self = this;
        // this.changeLanguage( defaultLanguage );
    }

    initDefaultLanguage( completeCllaback = null ){
        const self = this;
        self.changeBundle( self.getConfigBundleName(), completeCllaback );
    }

    isDefaultBundle(){
        return this._currBundleName == EBundleName.Default;
    }

    changeBundle( bundle: EBundleName, completeCallback = null ){
        const self = this;

        console.log( "I18Mgr changeLanguage " + bundle );
        if( self._currBundleName == bundle ){
            if( completeCallback ){
                completeCallback( null );
            }
            return;
        };

        self._currBundleName = bundle;
        self._mapSpriteFrame.clear();
        self._mapRes.clear();
        self._mapPrefabLoad.clear();
        
        if( self._currBundleName == EBundleName.Default ){
            // 有正在使用的情况,不能立即释放,切换语言包应当重启
            // if( self._currBundle != null ){
            //     self._currBundle.releaseAll();
            //     assetManager.removeBundle( self._currBundle );
            // }
            self._currBundle = null;
            
            console.log( "I18Mgr load resources" );
            const url = EBundleName.Default + "/" + TxtMgr.url;
            resources.load( url, JsonAsset, ( err: Error, data: JsonAsset )=>{
                if( err ){
                    console.error( `I18Mgr load ${url} error` );
                    console.error( err );

                    if( completeCallback ){
                        completeCallback( err );
                    }
                    return;
                }
                self.onLoadTxt( data );

                if( completeCallback ){
                    completeCallback( null );
                }
            } );
        }else{
            console.log( "I18Mgr loadBundle " + self._currBundleName );
            assetManager.loadBundle( self._currBundleName, (err: Error | null, bundle: AssetManager.Bundle)=>{
                if( err ){
                    console.log( `I18Mgr loadBundle ${self._currBundleName} error` );
                    console.error( err );
                    self.changeBundle( EBundleName.Default, completeCallback );
                    return;
                }
    
                // 有正在使用的情况,不能立即释放,切换语言包应当重启
                // if( self._currBundle != null ){
                //     self._currBundle.releaseAll();
                //     assetManager.removeBundle( self._currBundle );
                // }
                self._currBundle = bundle;
                bundle.load( TxtMgr.url, JsonAsset, ( err: Error, data: JsonAsset )=>{
                    if( err ){
                        console.error( `I18Mgr load ${TxtMgr.url} error` );
                        console.error( err );
                        
                        if( completeCallback ){
                            completeCallback( err );
                        }
                        return;
                    }
                    self.onLoadTxt( data );

                    if( completeCallback ){
                        completeCallback( null );
                    }
                } );
            } );
        }
    }


    onLoadTxt( data: JsonAsset ){
        TxtMgr.getInstance().setData( data );
        EventCenter.getInstance().fire( "I18BundleReady" );
    }

    isReady(){
        const self = this;
        return (self._currBundleName == EBundleName.Default) || (self._currBundle != null);
    }

    getCurrBundleName(){
        return this._currBundleName;
    }

    getCurrBundle() : AssetManager.Bundle {
        return this._currBundle;
    }

    getRes( name: string, assetType: any, callback:((err: Error | null, data: any) => void) ){
        const self = this;

        const map = self._mapRes.get( assetType );
        if( map != null ){
            const resData = map.get( name );
            if( resData != null ){
                callback( null, resData );
                return;
            }
        }

        if( self._currBundleName == EBundleName.Default ){
            resources.load( EBundleName.Default + "/" + name, assetType, ( err, data )=>{
                callback( err, data );
            } );
        }else{
            if( self._currBundle ){
                self._currBundle.load( name, assetType, ( err, data )=>{
                    if( err == null ){
                        let map = self._mapRes.get( assetType );
                        if( map == null ){
                            map = new Map();
                            self._mapRes.set( assetType, map );
                        }
                        map.set( name, data );
                    }
                    callback( err, data );
                } );
            }else{
                callback( new Error( self._currBundleName + "bundle is null" ), null );
            }
        }
    }

    getConfigBundleName(){
        const self = this;
        
        let language = "";
        //浏览器语言参数
        if( sys.isBrowser ){
            const params = self.getURLParam();
            if( params != null && params[ "l" ] != null ){
                language = params[ "l" ];
            }
        }
        if( language != "" && language != null ){
            console.log( "I18Mgr get language by browser " + language );
            return I18LanguageHelper.getBundleByStr( language.toUpperCase() );
        }

        language = localStorage.getItem( "language" );
        if( language != "" && language != null ){
            console.log( "I18Mgr get language by localStorage " + language );
            return I18LanguageHelper.getBundleByStr(language);
        }
        
        console.log( "I18Mgr get language by default " + EBundleName.Default );
        return EBundleName.Default;
    }

    getURLParam(){
        if( !sys.isBrowser ) return null;

        const arrStr = window.location.href.split( "?" );
        if( arrStr.length <= 1 ) return null;

        const strParams = arrStr[1];
        const arrParam = strParams.split( "&" );
        const params = {};
        for( const str of arrParam ){
            const arr = str.split( "=" );
            if( arr.length != 2 ) continue;
            params[ arr[0] ] = arr[1];
        }
        return params;
    }

    preloadScene (
        sceneName: string,
        bundle: AssetManager.Bundle,
        onProgress : (completedCount: number, totalCount: number, item: any) => void,
        onLoaded: Director.OnSceneLoaded,
    ): void {
        const self = this;
        let loadCountTmp = 0;
        let totalCountTmp = 1;

        if( bundle == null ){
            bundle = assetManager.bundles.find((bundle): boolean => !!bundle.getSceneInfo(sceneName));
        }
        
        if (bundle) {
            bundle.loadScene(
                sceneName,
                null,
                (finished: number, total: number, item: any)=>{
                    loadCountTmp = finished;
                    totalCountTmp = total;
                    onProgress( finished, total * (self._currBundleName == EBundleName.Default ? 1 : 2), item );
                },

                (err: Error | null, data: SceneAsset)=>{
                    self.loadResByNode( data.scene, onLoaded, ( finished: number, total: number )=>{
                        onProgress( loadCountTmp + finished, totalCountTmp + total, null );
                    } );
                }
            );
        } else {
            const err = `Can not preload the scene "${sceneName}" because it is not in the build settings.`;
            if (onLoaded) {
                onLoaded(new Error(err));
            }
            error(`preloadScene: ${err}`);
        }
    }

    loadScene(sceneName: string, onLaunched?: Director.OnSceneLaunched, onUnloaded?: Director.OnUnload): boolean {
        const self = this;
        const direct = director;
        if (direct[ "_loadingScene" ]) {
            warnID(1208, sceneName, direct[ "_loadingScene" ]);
            return false;
        }
        const bundle = assetManager.bundles.find((bundle): boolean => !!bundle.getSceneInfo(sceneName));
        if (bundle) {
            direct.emit( "director_before_scene_loading", sceneName);
            direct[ "_loadingScene" ] = sceneName;
            // eslint-disable-next-line no-console
            console.time(`LoadScene ${sceneName}`);
            bundle.loadScene(sceneName, (err, scene): void => {
                // eslint-disable-next-line no-console
                console.timeEnd(`LoadScene ${sceneName}`);
                direct[ "_loadingScene" ] = '';
                if (err) {
                    error(err);
                    if (onLaunched) {
                        onLaunched(err);
                    }
                } else {
                    self.loadResByNode( scene.scene, ()=>{
                        direct.runSceneImmediate(scene, onUnloaded, onLaunched);
                    } );
                }
            });
            return true;
        } else {
            errorID(1209, sceneName);
            return false;
        }
    }

    loadResByNode( loadNode: Node, 
        loadFinishedCallback:Function = null,
        progressCallback:Function = null 
    ){
        const self = this;
        if( self._currBundleName == EBundleName.Default ){
            if( loadFinishedCallback ){
                loadFinishedCallback();
            }
            return;
        }
        
        const mapResInfo = new Map<EI18AssetType, string[]>();
        self.getResInfoInNode( loadNode, mapResInfo );
        self.loadResByUrls( mapResInfo, loadFinishedCallback, progressCallback );
    }

    loadResByPrefab( prefab: Prefab, 
        loadFinishedCallback:Function = null,
        progressCallback:Function = null 
    ){
        const self = this;
        if( self._currBundleName == EBundleName.Default ){
            if( loadFinishedCallback ){
                loadFinishedCallback();
            }
            return;
        }

        const mapResInfo = new Map<EI18AssetType, string[]>();
        self.getResInfoInPrefab( prefab, mapResInfo );
        self.loadResByUrls( mapResInfo, loadFinishedCallback, progressCallback );
    }

    loadResByUrls( mapResInfo:Map<EI18AssetType, string[]>,
        loadFinishedCallback:Function = null,
        progressCallback:Function = null 
     ){
        if( mapResInfo.size == 0 ){
            if( loadFinishedCallback ){
                loadFinishedCallback();
            }
            return;
        }

        const self = this;

        let totalCount = 0;
        for( const resInfo of mapResInfo ){
            totalCount += resInfo[1].length;
        }

        let loadCount = 0;
        for( const resInfo of mapResInfo ){
            const resType = self.getAssetType( resInfo[0] );
            const szUrl = resInfo[1];
            for( const url of szUrl ){
                console.log( "i18Mgr loadResByUrls start " + url );
                const assetUrl = resInfo[0] == EI18AssetType.SpriteFrame ? url + "/spriteFrame" : url;
                self.getRes( assetUrl, resType, ( err, assetData : Asset )=>{
                    if( err ){
                        if( loadFinishedCallback ){
                            loadFinishedCallback( err );
                        }
                        return;
                    }
                    console.log( "i18Mgr loadResByUrls end " + assetData.name );
                    ++loadCount;
                    if( progressCallback ){
                        progressCallback( loadCount, totalCount );
                    }
                    if( loadCount >= totalCount ){
                        if( loadFinishedCallback ){
                            loadFinishedCallback( null );
                        }
                    }
                } );
            }
        }
    }

    getAssetType( assetType: EI18AssetType ){
        switch( assetType ){
            case EI18AssetType.SpriteFrame: return SpriteFrame;
            case EI18AssetType.SkeletonData: return sp.SkeletonData;
            default: return Asset;
        }
    }

    getResInfoInNode( node: Node, mapResInfo: Map<EI18AssetType, string[]> ){
        const self = this;

        let mapInfoTmp = self.getNodePreUrl( node );
        if( mapInfoTmp == null ){
            mapInfoTmp = new Map<EI18AssetType, string[]>();
        }

        const addUrl = function( type:any, url: string ){
            let szUrl = mapInfoTmp.get( type );
            if( szUrl == null ){
                szUrl = [];
                mapInfoTmp.set( type, szUrl );
            }
            if( szUrl.indexOf( url ) == -1 ){
                szUrl.push( url );
            }
        }

        const i18CmptInfo = self.getI18CmptInfo( node );
        if( i18CmptInfo ){
            addUrl( i18CmptInfo[0], i18CmptInfo[1] );
        }

        const i18BtnUrls = self.getBtnUrls( node );
        for( const url of i18BtnUrls ){
            addUrl( EI18AssetType.SpriteFrame, url );
        }

        for( const info of mapInfoTmp ){
            const type = info[0];
            const szUrlTmp = info[1];

            const mapRes = self._mapRes.get( type );
            if( mapRes == null ) continue;

            for( let i=0; i<szUrlTmp.length; ){
                const url = szUrlTmp[i];
                if( mapRes.get( url ) != null ){
                    szUrlTmp.splice( i, 1 );
                }else{
                    ++i;
                }
            }
        }

        for( const info of mapInfoTmp ){
            const type = info[0];
            const szUrlTmp = info[1];
            if( szUrlTmp.length == 0 ) continue;

            let szUrl = mapResInfo.get( type );
            if( szUrl == null ){
                szUrl = [];
                mapResInfo.set( type, szUrl );
            }

            for( const url of szUrlTmp ){
                if( szUrl.indexOf(url) == -1 ){
                    szUrl.push( url );
                }
            }
        }

        const cmpts = node.components;
        for( const cmpt of cmpts ){
            for( const k in cmpt ){
                const obj = cmpt[ k ];
                if( obj instanceof Array ){
                    for( const objTmp of obj ){
                        if( objTmp instanceof Prefab ){                            
                            self.getResInfoInPrefab( objTmp, mapResInfo );
                        }
                    }
                }else if( obj instanceof Prefab ){
                    self.getResInfoInPrefab( obj, mapResInfo );
                }
            }
        }

        const children = node.children;
        for( const n of children ){
            self.getResInfoInNode( n, mapResInfo );
        }
    }

    getResInfoInPrefab( prefab:Prefab, mapResInfo: Map<EI18AssetType, string[]> ){
        const self = this;
        if( self._mapPrefabLoad.get( prefab.uuid ) == true ){
            return;
        }

        self._mapPrefabLoad.set( prefab.uuid, true );
        this.getResInfoInNode( prefab.data, mapResInfo );
    }

    getNodePreUrl( node: Node ){
        if( node == null ) return null;

        const cmpt = node.getComponent( I18PreUrl );
        if( cmpt == null ) return null;
        
        return cmpt.getUrls();
    }

    getBtnUrls( node: Node ){
        if( node == null ) return [];

        const cmpt = node.getComponent( I18BtnUrl );
        if( cmpt == null ) return [];
        
        return cmpt.getUrls();
    }

    getI18CmptInfo( node: Node ){
        if( node == null ) return null;

        const cmpt = node.getComponent( "I18Cmpt" );
        if( cmpt == null ) return null;

        const sprite = node.getComponent( Sprite );
        if( sprite != null ){
            return [ EI18AssetType.SpriteFrame, cmpt[ "_key" ] ];
        }

        const spine = node.getComponent( sp.Skeleton );
        if( spine != null ){
            return [ EI18AssetType.SkeletonData, cmpt[ "_key" ] ];
        }

        return null;
    }

    static getMailStr( info ) :{ title:string, content:string } {
        const languageName = I18Mgr.getInstance().getCurrBundleName();
        if( info._lang != null && info._lang[ languageName ] != null ){
            return info._lang[ languageName ];
        }else{
            return {
                title: info._defTitle,
                content: info._defContent
            };
        }
   }
}