import { _decorator, Button, Component, Input, input, instantiate, Node, RigidBody, Slider, SphereCollider, v3, Vec3 } from 'cc'
import { Loader } from './Loader'
const { ccclass, property } = _decorator

const MAX_FORCE = 100

const MIN_MOVE_TIME = 1000 * 2

@ccclass('WhiteBall')
export class WhiteBall extends Component {
    @property(RigidBody)
    rigid: RigidBody

    @property(SphereCollider)
    collider: SphereCollider

    @property(Node)
    cue: Node

    @property(Node)
    button: Node

    @property(Node)
    line: Node

    // 冲量
    impulse: Vec3
    // 白球旋转角度
    angle = 0
    // 力
    force = MAX_FORCE / 2
    // 开始时位置
    cueStartPositionX: number
    // 球杆位置
    cuePosition: Vec3

    // 瞄准方向
    aimDir: Vec3

    v = new Vec3()

    hitTime: number
    dirFactor = 0

    start() {
        this.cueStartPositionX = this.cue.position.x
        this.cuePosition = this.cue.position
        this.rigid.useCCD = true


        input.on(Input.EventType.TOUCH_MOVE, (e) => {
            const deltaX = e.getDeltaX()
            const deltaY = e.getDeltaY()
            const max = Math.max(Math.abs(deltaX), Math.abs(deltaY))
            let delta
            if (Math.abs(deltaX) === max) {
                delta = deltaX
            } else {
                delta = deltaY
            }

            const next = this.dirFactor + delta * 0.001
            this.dirFactor = next > 1 ? next - 1 : next
            this.angle = 360 * (this.dirFactor - 0.5)
            this.node.angle = this.angle
        })
    }

    protected onLoad(): void {
        this.initBalls()
    }
    protected onDestroy(): void {
        input.off(Input.EventType.TOUCH_MOVE)
    }

    update(deltaTime: number) {
        if (Date.now() - this.hitTime > MIN_MOVE_TIME) {
            this.button.getComponent(Button).interactable = true
            this.line.active = true
            this.cue.active = true
            this.node.angle = this.angle
            this.rigid.clearState()
        }
    }

    async initBalls() {
        const ballPrefabs = await Loader.loadBalls()
        const nodes = ballPrefabs.map(instantiate) as Node[]

        const startX = -7.2
        const mat = [
            [1],
            [2, 3],
            [4, 5, 6],
            [7, 8, 9, 10],
            [11, 12, 13, 14, 15]
        ]

        const spaceRow = 1.1
        const spaceCol = 1.2
        mat.forEach((arr, row) => {
            const startY = -spaceCol * row / 2

            arr.forEach((n, col) => {
                const node = nodes[n - 1]
                node.position = v3(startX - spaceRow * row, startY + spaceCol * col, 0)
                node.getComponent(RigidBody).useCCD = true
                this.node.parent.addChild(node)
            })
        })
    }

    onHit() {
        this.impulse = this.node.getWorldPosition().subtract(this.cue.worldPosition).normalize().multiplyScalar(this.force)
        this.rigid.applyImpulse(this.impulse)

        this.cue.active = false
        this.line.active = false
        this.button.getComponent(Button).interactable = false

        this.hitTime = Date.now()
    }

    onDirChange(e: Slider) {
        this.angle = 360 * (e.progress - 0.5)
        this.node.angle = this.angle
    }

    onForceChange(e: Slider) {
        this.force = MAX_FORCE * e.progress + 10

        this.cuePosition.x = this.cueStartPositionX + 5 * e.progress
        this.cue.setPosition(this.cuePosition)
    }
}

