import { AssetManager, assetManager, director, Director, error, errorID, JsonAsset, Node, Prefab, resources, SceneAsset, Sprite, SpriteFrame, sys, warnID } from "cc";
import TxtMgr from "./TxtMgr";
import I18LanguageHelper, { EBundleName } from "./I18LanguageHelper";
import { I18PreUrl } from "./I18PreUrl";
import { dispatchEvent } from "../EventManager";
import I18Const from "./I18Const";


export default class I18Mgr {
    static instance : I18Mgr = null;
    static getInstance(): I18Mgr{
        if( I18Mgr.instance == null ){
            I18Mgr.instance = new I18Mgr();
        }
        return I18Mgr.instance;
    }

    _currBundleName : EBundleName = EBundleName.NONE;
    _currBundle : AssetManager.Bundle = null;

    _mapSpriteFrame = new Map();
    _mapPrefabLoad = new Map();

    constructor(){
        // const self = this;
        // this.changeLanguage( defaultLanguage );
    }

    initDefaultLanguage( completeCllaback = null ){
        const self = this;
        self.changeBundle( self.getConfigBundleName(), completeCllaback );
    }

    changeBundle( bundle: EBundleName, completeCallback = null ){
        const self = this;

        console.log( "I18Mgr changeLanguage " + bundle );
        if( self._currBundleName == bundle ){
            if( completeCallback ){
                completeCallback( null );
            }
            return;
        };

        self._currBundleName = bundle;
        self._mapSpriteFrame.clear();        
        self._mapPrefabLoad.clear();
        
        if( self._currBundleName == EBundleName.Default ){
            // 有正在使用的情况,不能立即释放,切换语言包应当重启 或者 释放后立即重新加载
            // if( self._currBundle != null ){
            //     self._currBundle.releaseAll();
            //     assetManager.removeBundle( self._currBundle );
            // }
            self._currBundle = null;
            
            console.log( "I18Mgr load resources" );
            const url = EBundleName.Default + "/" + I18Const.TXT_URL;
            resources.load( url, JsonAsset, ( err: Error, data: JsonAsset )=>{
                if( err ){
                    console.error( `I18Mgr load ${url} error` );
                    console.error( err );

                    if( completeCallback ){
                        completeCallback( err );
                    }
                    return;
                }
                self.onLoadTxt( data );

                if( completeCallback ){
                    completeCallback( null );
                }
            } );
        }else{
            console.log( "I18Mgr loadBundle " + self._currBundleName );
            assetManager.loadBundle( self._currBundleName, (err: Error | null, bundle: AssetManager.Bundle)=>{
                if( err ){
                    console.log( `I18Mgr loadBundle ${self._currBundleName} error` );
                    console.error( err );
                    self.changeBundle( EBundleName.Default, completeCallback );
                    return;
                }
    
                // 有正在使用的情况,不能立即释放,切换语言包应当重启 或者 释放后立即重新加载
                // if( self._currBundle != null ){
                //     self._currBundle.releaseAll();
                //     assetManager.removeBundle( self._currBundle );
                // }
                self._currBundle = bundle;
                bundle.load( I18Const.TXT_URL, JsonAsset, ( err: Error, data: JsonAsset )=>{
                    if( err ){
                        console.error( `I18Mgr load ${I18Const.TXT_URL} error` );
                        console.error( err );
                        
                        if( completeCallback ){
                            completeCallback( err );
                        }
                        return;
                    }
                    self.onLoadTxt( data );

                    if( completeCallback ){
                        completeCallback( null );
                    }
                } );
            } );
        }
    }


    onLoadTxt( data: JsonAsset ){
        TxtMgr.getInstance().setData( data );
        dispatchEvent( "I18BundleReady" );
    }

    isReady(){
        const self = this;
        return (self._currBundleName == EBundleName.Default) || (self._currBundle != null);
    }

    getCurrBundleName(){
        return this._currBundleName;
    }

    getCurrBundle() : AssetManager.Bundle {
        return this._currBundle;
    }

    getRes( name: string, assetType: any, callback:((err: Error | null, data: any) => void) ){
        const self = this;

        const resData = self._mapSpriteFrame.get( name );
        if( resData != null ){
            callback( null, resData );
            return;
        }

        if( self._currBundleName == EBundleName.Default ){
            resources.load( EBundleName.Default + "/" + name, assetType, ( err, data )=>{
                callback( err, data );
            } );
        }else{
            if( self._currBundle ){
                self._currBundle.load( name, assetType, ( err, data )=>{
                    if( err == null ){
                        self._mapSpriteFrame.set( name, data );
                    }
                    callback( err, data );
                } );
            }else{
                callback( new Error( self._currBundleName + "bundle is null" ), null );
            }
        }
    }

    getConfigBundleName(){
        const self = this;
        
        let language = "";
        //浏览器语言参数
        if( sys.isBrowser ){
            const params = self.getURLParam();
            if( params != null && params[ "language" ] != null ){
                language = params[ "language" ];
            }
        }
        if( language != "" && language != null ){
            console.log( "I18Mgr get language by browser " + language );
            return I18LanguageHelper.getBundleByStr(language);
        }

        //从他处获取语言参数
        
        console.log( "I18Mgr get language by default " + EBundleName.Default );
        return EBundleName.Default;
    }

    getURLParam(){
        if( !sys.isBrowser ) return null;

        const arrStr = window.location.href.split( "?" );
        if( arrStr.length <= 1 ) return null;

        const strParams = arrStr[1];
        const arrParam = strParams.split( "&" );
        const params = {};
        for( const str of arrParam ){
            const arr = str.split( "=" );
            if( arr.length != 2 ) continue;
            params[ arr[0] ] = arr[1];
        }
        return params;
    }

    preloadScene (
        sceneName: string,
        onProgress : (completedCount: number, totalCount: number, item: any) => void,
        onLoaded: Director.OnSceneLoaded,
    ): void {
        const self = this;
        let loadCountTmp = 0;
        let totalCountTmp = 1;
        const bundle = assetManager.bundles.find((bundle): boolean => !!bundle.getSceneInfo(sceneName));
        if (bundle) {
            bundle.loadScene(
                sceneName,
                null,
                (finished: number, total: number, item: any)=>{
                    loadCountTmp = finished;
                    totalCountTmp = total;
                    onProgress( finished, total * (self._currBundleName == EBundleName.Default ? 1 : 2), item );
                },

                (err: Error | null, data: SceneAsset)=>{
                    self.loadResByNode( data.scene, onLoaded, ( finished: number, total: number )=>{
                        onProgress( loadCountTmp + finished, totalCountTmp + total, null );
                    } );
                }
            );
        } else {
            const err = `Can not preload the scene "${sceneName}" because it is not in the build settings.`;
            if (onLoaded) {
                onLoaded(new Error(err));
            }
            error(`preloadScene: ${err}`);
        }
    }

    loadScene(sceneName: string, onLaunched?: Director.OnSceneLaunched, onUnloaded?: Director.OnUnload): boolean {
        const self = this;
        const direct = director;
        if (direct[ "_loadingScene" ]) {
            warnID(1208, sceneName, direct[ "_loadingScene" ]);
            return false;
        }
        const bundle = assetManager.bundles.find((bundle): boolean => !!bundle.getSceneInfo(sceneName));
        if (bundle) {
            direct.emit( "director_before_scene_loading", sceneName);
            direct[ "_loadingScene" ] = sceneName;
            // eslint-disable-next-line no-console
            console.time(`LoadScene ${sceneName}`);
            bundle.loadScene(sceneName, (err, scene): void => {
                // eslint-disable-next-line no-console
                console.timeEnd(`LoadScene ${sceneName}`);
                direct[ "_loadingScene" ] = '';
                if (err) {
                    error(err);
                    if (onLaunched) {
                        onLaunched(err);
                    }
                } else {
                    self.loadResByNode( scene.scene, ()=>{
                        direct.runSceneImmediate(scene, onUnloaded, onLaunched);
                    } );
                }
            });
            return true;
        } else {
            errorID(1209, sceneName);
            return false;
        }
    }

    loadResByNode( loadNode: Node, 
        loadFinishedCallback:Function = null,
        progressCallback:Function = null 
    ){
        const self = this;
        if( self._currBundleName == EBundleName.Default ){
            if( loadFinishedCallback ){
                loadFinishedCallback();
            }
            return;
        }
        
        const arrUrl = [];
        self.getSpriteUrlInNode( loadNode, arrUrl );
        self.loadResByUrls( arrUrl, loadFinishedCallback, progressCallback );
    }

    loadResByPrefab( prefab: Prefab, 
        loadFinishedCallback:Function = null,
        progressCallback:Function = null 
    ){
        const self = this;
        if( self._currBundleName == EBundleName.Default ){
            if( loadFinishedCallback ){
                loadFinishedCallback();
            }
            return;
        }

        const arrUrl = [];
        self.getSpriteUrlInPrefab( prefab, arrUrl );
        self.loadResByUrls( arrUrl, loadFinishedCallback, progressCallback );
    }

    loadResByUrls( arrUrl:Array<string>,
        loadFinishedCallback:Function = null,
        progressCallback:Function = null 
     ){
        if( arrUrl.length == 0 ){
            if( loadFinishedCallback ){
                loadFinishedCallback();
            }
            return;
        }

        const self = this;

        const totalCount = arrUrl.length;
        let loadCount = 0;

        for( const url of arrUrl ){
            self.getRes( url + "/spriteFrame", SpriteFrame, ( err, spriteFrame )=>{
                ++loadCount;
                if( progressCallback ){
                    progressCallback( loadCount, totalCount );
                }
                if( loadCount >= totalCount ){
                    if( loadFinishedCallback ){
                        loadFinishedCallback();
                    }
                }
            } );
        }
    }

    getSpriteUrlInNode( node: Node, arrUrl:Array<string> ){
        const self = this;

        const arrUrlTmp = self.getNodePreUrl( node );
        const i18Key = self.getSpriteI18Key( node );
        if( i18Key != null && i18Key != "" ){
            arrUrlTmp.push( i18Key );            
        }

        for( const url of arrUrlTmp ){
            if( self._mapSpriteFrame.get( url ) == null ){
                if( arrUrl.indexOf(url) == -1 ){
                    arrUrl.push( url );
                }
            }
        }

        const cmpts = node.components;
        for( const cmpt of cmpts ){
            for( const k in cmpt ){
                const obj = cmpt[ k ];
                if( obj instanceof Array ){
                    for( const objTmp of obj ){
                        if( objTmp instanceof Prefab ){                            
                            self.getSpriteUrlInPrefab( objTmp, arrUrl );
                        }
                    }
                }else if( obj instanceof Prefab ){
                    self.getSpriteUrlInPrefab( obj, arrUrl );
                }
            }
        }

        const children = node.children;
        for( const n of children ){
            self.getSpriteUrlInNode( n, arrUrl );
        }
    }

    getSpriteUrlInPrefab( prefab:Prefab, arrUrl:Array<string> ){
        const self = this;
        if( self._mapPrefabLoad.get( prefab.uuid ) == true ){
            return;
        }

        self._mapPrefabLoad.set( prefab.uuid, true );
        this.getSpriteUrlInNode( prefab.data, arrUrl );
    }

    getNodePreUrl( node: Node ){
        if( node == null ) return [];

        const cmpt = node.getComponent( I18PreUrl );
        if( cmpt == null ) return [];
        
        return cmpt.getUrls();
    }

    getSpriteI18Key( node: Node ){
        if( node == null ) return null;

        const cmpt = node.getComponent( "I18Cmpt" );
        if( cmpt == null ) return null;

        const sprite = node.getComponent( Sprite );
        if( sprite != null ) return cmpt[ "_key" ];

        return null;
    }
}