import { readFileSync } from "fs";

interface NodePath {
	name: string;
	path: string;
}

function parseNodeTree(jsonData: any[]): [NodePath[], string] {
	const nodeMap = new Map<number, { name: string, path: string }>();
	const result: NodePath[] = [];
	const prefabName = jsonData[0]._name;

	// 第一次遍历：建立节点映射
	jsonData.forEach((item, index) => {
		if (item.__type__ === "cc.Node") {
			nodeMap.set(index, {
				name: item._name,
				path: item._name
			});
		}
	});

	// 第二次遍历：建立父子关系
	jsonData.forEach((item, index) => {
		if (item.__type__ === "cc.Node" && item._parent) {
			const parentId = item._parent.__id__;
			const parent = nodeMap.get(parentId);
			const current = nodeMap.get(index);

			if (parent && current) {
				// 如果父节点是根节点（prefabName），则直接使用当前节点名称作为路径
				if (parent.name === prefabName) {
					current.path = current.name;
				} else {
					current.path = `${parent.path}/${current.name}`;
				}
			}
		}
	});

	// 收集所有以_开头的节点
	nodeMap.forEach((node) => {
		if (node.name.startsWith('_')) {
			let hasDuplicate = result.find(item => item.name === node.name);
			if (hasDuplicate) {
				console.log(`${node.name} 定义了重复的节点`);
				return;
			}
			result.push({
				name: node.name,
				path: node.path
			});
		}
	});

	return [result, prefabName];
}

export async function OnUiGenerate(isClass: boolean) {
	let selected = Editor.Selection.getSelected("asset");
	if (selected.length === 0) {
		console.log("请选择一个prefab");
		return;
	}

	let selectedPrefabUUID = selected[0];
	let path = await Editor.Message.request('asset-db', 'query-path', selectedPrefabUUID);
	if (path && path.endsWith(".prefab")) {
		let content = readFileSync(path, 'utf-8');
		let jsonData = JSON.parse(content);
		const [nodePaths, prefabName] = parseNodeTree(jsonData);
		let result = "";
		let code = generateUiCode(nodePaths);
		if (isClass) {
			result += `import { Node, Button, Label, ProgressBar, Sprite, SpriteFrame, sp } from "cc";\n`;
			let type = "Window";
			if (prefabName.includes("Window")) {
				type = "Window";
			} else if (prefabName.includes("Pop")) {
				type = "Pop";
			} else if (prefabName.includes("System")) {
				type = "System";
			} else if (prefabName.includes("Toast")) {
				type = "Toast";
			} else if (prefabName.includes("Widget")) {
				type = "Widget";
			}
			if (type == "Widget") {
				result += `import { UIWidget } from "db://assets/startup/framework/ui/UIWidget";\n`;
				result += `\n`;
				result += `export class ${prefabName} extends UIWidget {\n`;
				result += `\n`;
				result += code;
				result += `}\n`;
			} else {
				result += `import { UIBase } from "db://assets/startup/framework/ui/UIBase";\n`;
				result += `import { UIDecorator, UIType } from "db://assets/startup/framework/ui/UIDecorator";\n`;
				result += `\n`;
				result += `@UIDecorator(UIType.${type}, "请填写路径！！！！！！！！！！！！！")\n`;
				result += `export class ${prefabName} extends UIBase {\n`;
				result += `\n`;
				result += code;
				result += `}\n`;
			}

		} else {
			result += code;
		}
		console.log(result);
		console.log(Editor.Clipboard.write("text", result));
	} else {
		console.log("非法资源");
	}
}

// 根据前缀匹配生成UI脚本代码
function generateUiCode(nodePaths: NodePath[]): string {
	let result = "	/**\n";
	result += `	* 自动生成\n`;
	result += "	*/";
	result += "	\n";
	let fieldConent = "";
	let findContent = "	public onScriptGenerate(): void {\n";
	let eventContent = "";
	for (let nodePath of nodePaths) {
		if (nodePath.name.startsWith("_nd")) {
			fieldConent += `	private ${nodePath.name}: Node;\n`;
			findContent += `		this.${nodePath.name} = this.findNode("${nodePath.path}");\n`;
		} else if (nodePath.name.startsWith("_btn")) {
			fieldConent += `	private ${nodePath.name}: Button;\n`;
			findContent += `		this.${nodePath.name} = this.findComponent("${nodePath.path}",Button);\n`;
			eventContent += `		this.${nodePath.name}.node.on(Button.EventType.CLICK, this.on${nodePath.name}Click, this);\n`;
		} else if (nodePath.name.startsWith("_lb")) {
			fieldConent += `	private ${nodePath.name}: Label;\n`;
			findContent += `		this.${nodePath.name} = this.findComponent("${nodePath.path}",Label);\n`;
		} else if (nodePath.name.startsWith("_spr")) {
			fieldConent += `	private ${nodePath.name}: Sprite;\n`;
			findContent += `		this.${nodePath.name} = this.findComponent("${nodePath.path}",Sprite);\n`;
		} else if (nodePath.name.startsWith("_pb")) {
			fieldConent += `	private ${nodePath.name}: ProgressBar;\n`;
			findContent += `		this.${nodePath.name} = this.findComponent("${nodePath.path}",ProgressBar);\n`;
		}else if (nodePath.name.startsWith("_spi")) {
			fieldConent += `	private ${nodePath.name}: sp.Skeleton;\n`;
			findContent += `		this.${nodePath.name} = this.findComponent("${nodePath.path}",sp.Skeleton);\n`;
		}
	}
	findContent += eventContent;
	findContent += "	}\n";
	result += `${fieldConent}\n`;
	result += `${findContent}\n`;
	result += "	/*请不要修改此代码，否则会在下次生成时被覆盖*/";
	result += "\n";
	return result;
}
