import { AssetManager, JsonAsset, Node, Prefab, assetManager, director, game, resources, warn } from "cc";

type PresetConfig = { value: string; options: ReadonlyArray<{ value: string }> };

export default class PresetConfigManager {
    public static readonly presetBundleName = "presets";
    private static _bundle: AssetManager.Bundle;
    public static get bundle() { return PresetConfigManager._bundle; }
    private static _presetConfig: PresetConfig;
    public static get presetConfig() { return PresetConfigManager._presetConfig; }

    public static async initialize(config: JsonAsset): Promise<void> {
        const data = config.json as PresetConfig;

        PresetConfigManager._presetConfig = data;

        PresetConfigManager._bundle = await new Promise<AssetManager.Bundle>(resolve => assetManager.loadBundle(PresetConfigManager.presetBundleName, (error: Error | null, data: AssetManager.Bundle) => {
            if (error != null) return warn(error);
            resolve(data);
        }));
        PresetConfigManager.registerPipeline();
    }

    private static replacePresetAsset(task: AssetManager.Task) {
        const input = task.output = task.input;
        const bundle = PresetConfigManager.bundle;
        const preset = PresetConfigManager.presetConfig;
        for (let i = 0; i < input.length; i++) {
            const item = input[i] as AssetManager.RequestItem & { config: any, overrideUuid: string };
            const config = item.config;
            // 如果不是preset bundle的资源，则跳过
            if (config == null || config.name != PresetConfigManager.presetBundleName) continue;
            // @ts-ignore
            const path: string = item.info?.path;
            if (path == null) continue;

            const dirnames = path.split("/");
            if (dirnames.length < 2) continue;
            const optionName = dirnames[0];
            // 如果是加载同一个选项的内容，则跳过
            if (preset.value == optionName) continue;

            // 换成新路径
            const newPath = path.replace(optionName, preset.value);
            const info = bundle.getInfoWithPath(newPath);
            if (info == null) return console.warn(`preset asset ${newPath} is not exist.`)

            item.info = info;
            item.overrideUuid = info.uuid;
        }
    }

    private static createNodeWithPrefabBefore(node: Node): void {
        // node.prefab.fileId
    }

    protected static registerPipeline(): void {
        assetManager.transformPipeline.insert(PresetConfigManager.replacePresetAsset, 2);
        // hookScenePresetFileId
        Prefab._utils.createNodeWithPrefab = PresetConfigManager.createNodeWithPrefabBefore;
    }
}

game.onPostProjectInitDelegate.add(async () => {
    const config = await new Promise<JsonAsset>(resolve => resources.load<JsonAsset>("presetConfig", (error, data) => resolve(data)));
    PresetConfigManager.initialize(config);
})