import { AssetManager, assetManager, director, Director, error, errorID, JsonAsset, Node, Prefab, resources, Sprite, SpriteFrame, sys, warnID, sp } from "cc";
import TxtMgr from "./TxtMgr";
import { dispatchEvent } from "./EventManager";


export default class I18Mgr {
    static instance : I18Mgr = null;
    static getInstance(): I18Mgr{
        if( I18Mgr.instance == null ){
            I18Mgr.instance = new I18Mgr();
        }
        return I18Mgr.instance;
    }
    static defaultLanguage = "CN";

    _currLanguage = "";
    _currBundle : AssetManager.Bundle = null;

    _mapSpriteFrame = new Map();
    _mapPrefabLoad = new Map();

    constructor(){
        // const self = this;
        // this.changeLanguage( defaultLanguage );
    }

    initDefaultLanguage(){
        const self = this;
        self.changeLanguage( self.getInitLanguage() );
    }

    changeLanguage( lang: string, completeCallback = null ){
        const self = this;
        if( self._currLanguage == lang ){
            if( completeCallback ){
                completeCallback( null );
            }
            return;
        };

        self._currLanguage = lang;
        self._mapSpriteFrame.clear();        
        self._mapPrefabLoad.clear();
        
        if( self._currLanguage == I18Mgr.defaultLanguage ){
            const url = I18Mgr.defaultLanguage + "/" + TxtMgr.url;
            resources.load( url, JsonAsset, ( err: Error, data: JsonAsset )=>{
                if( err ){
                    error( `I18Mgr load ${url} error` );
                    error( err );
                    completeCallback( err );
                    return;
                }
                self.onLoadTxt( data );

                if( completeCallback ){
                    completeCallback( null );
                }
            } );
        }else{
            assetManager.loadBundle( self._currLanguage, (err: Error | null, bundle: AssetManager.Bundle)=>{
                if( err ){
                    self._currLanguage = I18Mgr.defaultLanguage;
                    error( `I18Mgr loadBundle ${self._currLanguage} error` );
                    error( err );
                    completeCallback( err );
                    return;
                }
    
                self._currBundle = bundle;
                bundle.load( TxtMgr.url, JsonAsset, ( err: Error, data: JsonAsset )=>{
                    if( err ){
                        error( `I18Mgr load ${TxtMgr.url} error` );
                        error( err );
                        completeCallback( err );
                        return;
                    }
                    self.onLoadTxt( data );

                    if( completeCallback ){
                        completeCallback( null );
                    }
                } );
            } );
        }
    }

    onLoadTxt( data: JsonAsset ){
        TxtMgr.getInstance().setData( data );
        dispatchEvent( "I18BundleReady" );
    }

    isReady(){
        const self = this;
        return (self._currLanguage == I18Mgr.defaultLanguage) || (self._currBundle != null);
    }

    getCurrLanguage(){
        return this._currLanguage;
    }

    getCurrBundle() : AssetManager.Bundle {
        return this._currBundle;
    }

    getRes( name: string, assetType: any, callback:((err: Error | null, data: any) => void) ){
        const self = this;

        const resData = self._mapSpriteFrame.get( name );
        if( resData != null ){
            callback( null, resData );
            return;
        }

        if( self._currLanguage == I18Mgr.defaultLanguage ){
            resources.load( I18Mgr.defaultLanguage + "/" + name, assetType, ( err, data )=>{
                if( err == null ){
                    self._mapSpriteFrame.set( I18Mgr.defaultLanguage + "/" + name, data );
                }
                callback( err, data );
            } );
        }else{
            if( self._currBundle ){
                self._currBundle.load( name, assetType, ( err, data )=>{
                    if( err == null ){
                        self._mapSpriteFrame.set( name, data );
                    }
                    callback( err, data );
                } );
            }else{
                callback( new Error( self._currLanguage + "bundle is null" ), null );
            }
        }
    }

    getInitLanguage(){
        const self = this;
        
        let language = "";

        //浏览器语言参数
        if( sys.isBrowser ){
            const params = self.getURLParam();
            if( params != null && params[ "language" ] != null ){
                language = params[ "language" ];
            }
        }
        if( language != "" && language != null ){
            return language;
        }
        
        return I18Mgr.defaultLanguage;
    }

    getURLParam(){
        if( !sys.isBrowser ) return null;

        const arrStr = window.location.href.split( "?" );
        if( arrStr.length <= 1 ) return null;

        const strParams = arrStr[1];
        const arrParam = strParams.split( "&" );
        const params = {};
        for( const str of arrParam ){
            const arr = str.split( "=" );
            if( arr.length != 2 ) continue;
            params[ arr[0] ] = arr[1];
        }
        return params;
    }

    loadScene(sceneName: string, onLaunched?: Director.OnSceneLaunched, onUnloaded?: Director.OnUnload): boolean {
        const self = this;
        const direct = director;
        if (direct[ "_loadingScene" ]) {
            warnID(1208, sceneName, direct[ "_loadingScene" ]);
            return false;
        }
        const bundle = assetManager.bundles.find((bundle): boolean => !!bundle.getSceneInfo(sceneName));
        if (bundle) {
            direct.emit( "director_before_scene_loading", sceneName);
            direct[ "_loadingScene" ] = sceneName;
            // eslint-disable-next-line no-console
            console.time(`LoadScene ${sceneName}`);
            bundle.loadScene(sceneName, (err, scene): void => {
                // eslint-disable-next-line no-console
                console.timeEnd(`LoadScene ${sceneName}`);
                direct[ "_loadingScene" ] = '';
                if (err) {
                    error(err);
                    if (onLaunched) {
                        onLaunched(err);
                    }
                } else {
                    self.loadResByNode( scene.scene, ()=>{
                        direct.runSceneImmediate(scene, onUnloaded, onLaunched);
                    } );
                }
            });
            return true;
        } else {
            errorID(1209, sceneName);
            return false;
        }
    }

    loadResByNode( loadNode: Node, loadFinishedCallback:Function = null ){
        const self = this;

        if( self._currLanguage == I18Mgr.defaultLanguage ){
            if( loadFinishedCallback ){
                loadFinishedCallback();
            }
            return;
        }

        const arrKey = [];
        self.getSpriteI18KeyInNode( loadNode, arrKey );

        const totalCount = arrKey.length;
        let loadCount = 0;

        for( const url of arrKey ){
            self.getRes( url + "/spriteFrame", SpriteFrame, ( err, spriteFrame )=>{
                ++loadCount;
                if( loadCount >= totalCount ){
                    if( loadFinishedCallback ){
                        loadFinishedCallback();
                    }
                }
            } );
        }
    }

    getSpriteI18KeyInNode( node: Node, arrKey:Array<string> ){
        const self = this;

        const i18Key = self.getSpriteI18Key( node );
        if( i18Key != null && i18Key != "" ){
            if( self._mapSpriteFrame.get( i18Key ) == null ){
                if( arrKey.indexOf(i18Key) == -1 ){
                    arrKey.push( i18Key );
                }
            }
        }

        const cmpts = node.components;
        for( const cmpt of cmpts ){
            for( const k in cmpt ){
                const obj = cmpt[ k ];
                if( obj instanceof Array ){
                    for( const objTmp of obj ){
                        if( objTmp instanceof Prefab ){                            
                            self.getSpriteI18KeyInPrefab( objTmp, arrKey );
                        }
                    }
                }else if( obj instanceof Prefab ){
                    self.getSpriteI18KeyInPrefab( obj, arrKey );
                }
            }
        }

        const children = node.children;
        for( const n of children ){
            self.getSpriteI18KeyInNode( n, arrKey );
        }
    }

    getSpriteI18KeyInPrefab( prefab:Prefab, arrKey:Array<string> ){
        const self = this;
        if( self._mapPrefabLoad.get( prefab.uuid ) == true ){
            return;
        }

        self._mapPrefabLoad.set( prefab.uuid, true );
        this.getSpriteI18KeyInNode( prefab.data, arrKey );
    }

    getSpriteI18Key( node: Node ){
        if( node == null ) return null;

        const cmpt = node.getComponent( "I18Cmpt" );
        if( cmpt == null ) return null;

        const sprite = node.getComponent( Sprite );
        if( sprite != null ) return cmpt[ "_key" ];

        return null;
    }



}