/*
 * @Author: 林武
 * @Date: 2022-06-01 10:28:43
 * @LastEditors: 林武
 * @LastEditTime: 2022-06-09 10:24:18
 * @FilePath: \QuadTree\assets\Script\CollisionQuadTree.ts
 * @Description: 
 * 
 * Copyright (c) 2022 by 林武/长沙游品, All Rights Reserved. 
 */

import { Quadtree } from "./QuadTree";

const { ccclass, property } = cc._decorator;

@ccclass
export default class CollisionQuadTree extends cc.Component {

    @property({ type: cc.Graphics, tooltip: CC_DEV && '绘图组件' })
    protected graphics: cc.Graphics = null;

    @property({ type: cc.Prefab, tooltip: CC_DEV && '添加物体节点' })
    protected item: cc.Prefab = null;

    @property({ type: cc.Label, tooltip: CC_DEV && '按钮文字' })
    protected btnTreeLabel: cc.Label = null;

    @property({ type: cc.Label, tooltip: CC_DEV && '场景节点数量' })
    protected ItemNum: cc.Label = null;

    @property({ displayName: CC_DEV && '是否启用四叉树' })
    protected openQuadTree: boolean = true;


    // LIFE-CYCLE CALLBACKS:

    /** 生产的节点 */
    private nodes: Array<cc.Node> = [];

    /** 四叉树 */
    protected quadTree: Quadtree = null;

    /** 生命周期：加载 */
    onLoad() {

    }

    /** 生命周期：节点开始 */
    start() {
        this.btnTreeLabel.string = this.openQuadTree ? "关闭四叉树" : "开启四叉树";
        this.initQuadTree();
    }

    /** 初始化四叉树 */
    protected initQuadTree() {
        var bounds = {
            x: 0,
            y: 0,
            width: cc.Canvas.instance.node.width,
            height: cc.Canvas.instance.node.height
        }
        this.quadTree = new Quadtree(bounds);

        for (let i = 0; i < 500; i++) {
            let newNode = cc.instantiate(this.item);
            this.node.addChild(newNode);
            this.nodes.push(newNode)
            this.quadTree.insert(newNode);
        }

        this.ItemNum.string = "当前节点数量：" + this.nodes.length;
    }

    /** 切换四叉树状态按钮 */
    clickBtnTree() {
        this.openQuadTree = !this.openQuadTree;
        this.btnTreeLabel.string = this.openQuadTree ? "关闭四叉树" : "开启四叉树";

        if (!this.openQuadTree) {
            this.drawQuadTreeNodes();
        }
    }

    /** 增加节点数量按钮 */
    clickAddBtn() {
        for (let i = 0; i < 50; i++) {
            let newNode = cc.instantiate(this.item);
            this.node.addChild(newNode);
            this.nodes.push(newNode)
            this.quadTree.insert(newNode);
        }
        this.ItemNum.string = "当前节点数量：" + this.nodes.length;
    }

    /** 减少节点数量按钮 */
    clickSubtractBtn() {
        for (let i = 0; i < 50; i++) {
            let newNode = this.nodes[this.nodes.length - 1] as cc.Node;
            this.nodes.splice(this.nodes.length - 1, 1);
            newNode.destroy();
        }
        this.quadTree.clear();
        this.ItemNum.string = "当前节点数量：" + this.nodes.length;
    }

    /** 普通碰撞检测 */
    normalCheck() {
        for (let node of this.nodes) {
            node.getComponent("item").setIsCollision(false)
        }
        for (let i = 0; i < this.nodes.length; i++) {
            let node = this.nodes[i];
            for (let j = 0; j < this.nodes.length; j++) {
                let targetNode = this.nodes[j];
                if (targetNode === node) continue;
                let isCollision: any = targetNode.getBoundingBoxToWorld().intersects(node.getBoundingBoxToWorld());
                if (isCollision) {
                    node.getComponent("item").setIsCollision(isCollision);
                    targetNode.getComponent("item").setIsCollision(isCollision);
                }
            }
        }
    }

    /** 四叉树碰撞检测 */
    quadTreeCheck() {
        for (let node of this.nodes) {
            node.getComponent("item").setIsCollision(false);
            this.quadTree.insert(node);
        }

        // 绘制四叉树节点
        this.drawQuadTreeNodes();

        for (let i = 0; i < this.nodes.length; i++) {
            let node = this.nodes[i] as cc.Node;
            let targetNodes = this.quadTree.retrieve(node);
            for (let j = 0; j < targetNodes.length; j++) {
                let targetNode = targetNodes[j];
                if (targetNode === node) continue;
                let isCollision: any = targetNode.getBoundingBoxToWorld().intersects(node.getBoundingBoxToWorld());
                if (isCollision) {
                    node.getComponent("item").setIsCollision(isCollision);
                    targetNode.getComponent("item").setIsCollision(isCollision);
                }
            }
        }
        this.quadTree.clear();
    }

    /** 绘制四叉树节点 */
    protected drawQuadTreeNodes() {
        // 更新绘图节点位置
        this.graphics.getComponent(cc.Widget).updateAlignment();
        // 清除旧路径
        const graphics = this.graphics;
        graphics.clear();
        graphics.strokeColor = cc.color(0, 255, 0, 150);
        // 递归函数
        function creatPath(tree: Quadtree) {
            const subTrees = tree.nodes;
            // 是否有子节点？
            // 没有子节点才绘制路径
            if (subTrees.length === 0) {
                const rect = tree.bounds;
                graphics.rect(rect.x, rect.y, rect.width, rect.height);
            } else {
                // 递归子节点
                for (let i = 0; i < subTrees.length; i++) {
                    creatPath(subTrees[i]);
                }
            }
        }
        // 递归生成路径
        creatPath(this.quadTree);
        // 绘制路径
        graphics.stroke();
    }

    update(dt) {
        if (this.openQuadTree) {
            this.quadTreeCheck();
        } else {
            this.normalCheck();
        }
    }
}
