import { _decorator, Component, Node, Sprite, SpriteFrame, Button, ScrollView, UITransform, Layout, instantiate, assetManager, Prefab, resources } from 'cc';
const { ccclass, property } = _decorator;

// 定义资源类型枚举
enum ResourceType {
    BACKGROUND = 'background',
    ITEM = 'item',
    ALL = 'all'
}

@ccclass('WarehouseManager')
export class WarehouseManager extends Component {
    // 按钮布局节点引用
    @property({type: Node})
    private btnLayout: Node = null;
    
    // ScrollView组件引用
    @property({type: ScrollView})
    private scrollView: ScrollView = null;
    
    // 内容容器节点引用
    @property({type: Node})
    private content: Node = null;
    
    // 项目预制体引用 - 修正为Prefab类型
    @property({type: Prefab})
    private itemPrefab: Prefab = null;
    
    // 存储所有资源的SpriteFrame
    private backgroundResources: SpriteFrame[] = [];
    private itemResources: SpriteFrame[] = [];
    
    // 当前显示的资源类型
    private currentType: ResourceType = ResourceType.ALL;

    protected onLoad(): void {
        // 初始化按钮事件
        this.initButtons();
        
        // 加载所有资源
        this.loadAllResources();
    }

    /**
     * 初始化所有按钮点击事件
     */
    private initButtons(): void {
        // 获取所有按钮并绑定事件
        const bgBtn = this.btnLayout.getChildByName('bg')?.getComponent(Button);
        const djBtn = this.btnLayout.getChildByName('dj')?.getComponent(Button);
        const allBtn = this.btnLayout.getChildByName('all')?.getComponent(Button);
        
        if (bgBtn) {
            bgBtn.node.on(Button.EventType.CLICK, this.onBgBtnClick, this);
        }
        
        if (djBtn) {
            djBtn.node.on(Button.EventType.CLICK, this.onDjBtnClick, this);
        }
        
        if (allBtn) {
            allBtn.node.on(Button.EventType.CLICK, this.onAllBtnClick, this);
        }
    }

    /**
     * 加载所有资源
     */
    private async loadAllResources(): Promise<void> {
        try {
            // 加载背景资源 - 修正资源路径
            const bg1 = await this.loadSpriteFrame('backgrounds/fj_1_bg/spriteFrame');
            const bg2 = await this.loadSpriteFrame('backgrounds/fj_2_bg/spriteFrame');
            if (bg1) this.backgroundResources.push(bg1);
            if (bg2) this.backgroundResources.push(bg2);
            
            // 加载道具资源 - 修正资源路径
            const item1 = await this.loadSpriteFrame('common_battle/beiji/spriteFrame');
            const item2 = await this.loadSpriteFrame('common_battle/dingshen/spriteFrame');
            if (item1) this.itemResources.push(item1);
            if (item2) this.itemResources.push(item2);
            
            // 默认显示全部
            this.updateContent();
        } catch (error) {
            console.error('资源加载失败:', error);
        }
    }

    /**
     * 加载SpriteFrame资源
     * @param path 资源路径
     * @returns Promise<SpriteFrame | null>
     */
    private loadSpriteFrame(path: string): Promise<SpriteFrame | null> {
        return new Promise((resolve) => {
            resources.load(path, SpriteFrame, (err, spriteFrame) => {
                if (err) {
                    console.error(`加载资源 ${path} 失败:`, err);
                    resolve(null);
                    return;
                }
                resolve(spriteFrame);
            });
        });
    }

    /**
     * 背景按钮点击事件
     */
    private onBgBtnClick(): void {
        this.currentType = ResourceType.BACKGROUND;
        this.updateContent();
    }

    /**
     * 道具按钮点击事件
     */
    private onDjBtnClick(): void {
        this.currentType = ResourceType.ITEM;
        this.updateContent();
    }

    /**
     * 全部按钮点击事件
     */
    private onAllBtnClick(): void {
        this.currentType = ResourceType.ALL;
        this.updateContent();
    }

    /**
     * 更新ScrollView内容
     */
    private updateContent(): void {
        // 清空现有内容
        this.content.removeAllChildren();
        
        // 根据当前类型获取要显示的资源
        let resourcesToShow: SpriteFrame[] = [];
        
        switch (this.currentType) {
            case ResourceType.BACKGROUND:
                resourcesToShow = this.backgroundResources;
                break;
            case ResourceType.ITEM:
                resourcesToShow = this.itemResources;
                break;
            case ResourceType.ALL:
            default:
                resourcesToShow = [...this.backgroundResources, ...this.itemResources];
                break;
        }
        
        // 创建并添加新的项目
        resourcesToShow.forEach((spriteFrame) => {
            if (!spriteFrame) return;
            
            const item = instantiate(this.itemPrefab);
            const button = item.getComponent(Button);
            const sprite = item.getComponent(Sprite);
            
            if (sprite && spriteFrame) {
                sprite.spriteFrame = spriteFrame;
            }
            
            // 添加点击事件
            if (button) {
                button.node.on(Button.EventType.CLICK, () => {
                    this.onItemClick(spriteFrame);
                }, this);
            }
            
            this.content.addChild(item);
        });
        
        // 更新布局
        const layout = this.content.getComponent(Layout);
        if (layout) {
            layout.updateLayout();
        }
        
        // 重置ScrollView位置到顶部
        this.scrollView.scrollToTop();
    }

    /**
     * 项目点击事件
     * @param spriteFrame 点击的项目的SpriteFrame
     */
    private onItemClick(spriteFrame: SpriteFrame): void {
        console.log('点击了项目:', spriteFrame.name);
        // 这里可以添加点击后的逻辑，比如显示大图、使用道具等
    }

    protected onDestroy(): void {
        // 清理按钮事件
        const bgBtn = this.btnLayout.getChildByName('bg')?.getComponent(Button);
        const djBtn = this.btnLayout.getChildByName('dj')?.getComponent(Button);
        const allBtn = this.btnLayout.getChildByName('all')?.getComponent(Button);
        
        if (bgBtn) {
            bgBtn.node.off(Button.EventType.CLICK, this.onBgBtnClick, this);
        }
        
        if (djBtn) {
            djBtn.node.off(Button.EventType.CLICK, this.onDjBtnClick, this);
        }
        
        if (allBtn) {
            allBtn.node.off(Button.EventType.CLICK, this.onAllBtnClick, this);
        }
    }
}