import { assert, clamp, geometry, gfx,  Material, renderer, rendering, Vec4} from 'cc';
import { CameraConfigs, getPingPongRenderTarget, PipelineConfigs, PipelineContext } from './builtin-pipeline-custom';
import { makePipelineSettings, PipelineSettings, } from './builtin-pipeline-types-custom';


const { AABB, Sphere, intersect } = geometry;
const { ClearFlagBit, Color, Format, FormatFeatureBit, LoadOp, StoreOp, TextureType, Viewport } = gfx;
const { scene } = renderer;
const { CameraUsage, CSMLevel, LightType } = scene;
const sClearColorTransparentBlack = new Color(0, 0, 0, 0);

export interface ScreenNormalOutline {
    enabled: boolean;
    material: Material | null;
}


export function makeScreenNormalOutline(): ScreenNormalOutline {
    return {
        enabled: false,
        material: null
    };
}

export function fillRequiredScreenNormalOutline(value: ScreenNormalOutline): void {
    if (value.enabled === undefined) {
        value.enabled = false;
    }
    if (value.material === undefined) {
        value.material = null;
    }
}


export interface ScreenNormalOutlinePassConfigs {
    enabledOutline: boolean;
}

export class ScreenNormalOutlinePassBuilder implements rendering.PipelinePassBuilder {
    getConfigOrder(): number {
        return 0;
    }
    getRenderOrder(): number {
        return 600;
    }
    configCamera(
        camera: Readonly<renderer.scene.Camera>,
        pplConfigs: Readonly<PipelineConfigs>,
        cameraConfigs: CameraConfigs & ScreenNormalOutlinePassConfigs): void {
        cameraConfigs.enabledOutline = cameraConfigs.settings.outline.enabled
            && !!cameraConfigs.settings.outline.material;

        if (cameraConfigs.enabledOutline) {
            ++cameraConfigs.remainingPasses;
        }
    }
    setup(
        ppl: rendering.BasicPipeline,
        pplConfigs: Readonly<PipelineConfigs>,
        cameraConfigs: CameraConfigs & ScreenNormalOutlinePassConfigs,
        camera: renderer.scene.Camera,
        context: PipelineContext,
        prevRenderPass?: rendering.BasicRenderPassBuilder)
        : rendering.BasicRenderPassBuilder | undefined {
        if (!cameraConfigs.enabledOutline) {
            return prevRenderPass;
        }
        --cameraConfigs.remainingPasses;

        const inputColorName = context.colorName;
        const outputColorName
            = cameraConfigs.remainingPasses === 0
                ? cameraConfigs.colorName
                : getPingPongRenderTarget(context.colorName, 'UiColor', cameraConfigs.renderWindowId);
        context.colorName = outputColorName;
        const depthTextureName = context.depthStencilName;

        assert(!!cameraConfigs.settings.outline.material);
        return this._addPass(ppl, pplConfigs, cameraConfigs,
            cameraConfigs.settings,
            cameraConfigs.settings.outline.material,
            cameraConfigs.renderWindowId,
            cameraConfigs.width,
            cameraConfigs.height,
            inputColorName,
            cameraConfigs.nativeWidth,
            cameraConfigs.nativeHeight,
            outputColorName,
            depthTextureName);
    }
    private _addPass(
        ppl: rendering.BasicPipeline,
        pplConfigs: Readonly<PipelineConfigs>,
        cameraConfigs: CameraConfigs & ScreenNormalOutlinePassConfigs,
        settings: PipelineSettings,
        mat: Material,
        id: number,
        width: number,
        height: number,
        inputColorName: string,
        nativeWidth: number,
        nativeHeight: number,
        outputColorName: string,
        depthTextureName?: string,
    ): rendering.BasicRenderPassBuilder {
        const pass = ppl.addRenderPass(nativeWidth, nativeHeight, 'screen-normal-outline');
        pass.addRenderTarget(outputColorName, LoadOp.CLEAR, StoreOp.STORE, sClearColorTransparentBlack);
        pass.addTexture(inputColorName, 'outputResultMap');
        if (depthTextureName) {
            pass.addTexture(depthTextureName, 'depthTexture');
        }
        pass.addQueue(rendering.QueueHint.OPAQUE)
            .addFullscreenQuad(mat, 0);
        return pass;
    }
}