/**
 * 加载资源的静态类
 * 加载预制啥的
 */
import { Asset, AudioClip, Node, ParticleAsset, Prefab, Vec3, instantiate, resources } from "cc";
type Constructor<T = unknown> = new (...args: any[]) => T;
type AssetType<T = Asset> = Constructor<T>;
type LoadCompleteCallback<T> = (error: Error | null, asset: T) => void;


export default class ResourceUtil {

	/**
	 * 这里我们做一个缓存管理 用自带的去获取 注意这里我们的缓存是已经加载完成的资源
	 * 加载resource目录下的资源
	 * @param url 
	 * @param type 
	 * @param cb 
	 */
	private static loadResourcesBundle<T extends Asset>(url: string, type: AssetType<T> | null, cb: LoadCompleteCallback<T>) {
		let findRes = resources.get(url, type)
		if (findRes) {
			return cb(null, findRes as T);
		}
		if (type) {
			resources.load(url, type, (err, res) => {
				if (err) {
					if (cb) {
						cb(err, res);
					}
					return;
				}
				if (cb) {
					cb(err, res);
				}
			});
		} else {
			resources.load(url, (err, res) => {
				if (err) {
					if (cb) {
						cb(err, res as T);
					}
					return;
				}
				if (cb) {
					cb(err, res as T);
				}
			});
		}
	}

	/**
	 * 加载一个预制 并添加到父节点上
	 * @param path 
	 * @param parent 
	 * @param cb 
	 */
	public static createUI(path: string, parent: Node, cb?: (err: Error | null, node?: Node) => void, pos?: Vec3) {
		this.loadResourcesBundle("prefab/" + path, Prefab, (err: Error | null, prefab?: Prefab) => {
			if (err) {
				cb && cb(err, null)
				return;
			}
			const node = instantiate(prefab!);
			pos && node.setPosition(pos);
			parent?.addChild(node);
			if (cb) cb(null, node);
		});
	}

	/**
	 * 加载音乐资源 包括bgm和音效
	 */
	public static loadMusic(path: string, cb?: (err: Error | null, asset: AudioClip) => void) {
		this.loadResourcesBundle(path, AudioClip, cb)
	}

	/**
	 * 加载粒子资源
	 */
	public static loadParticle(path: string, cb?: (err: Error | null, asset: ParticleAsset) => void) {
		this.loadResourcesBundle("particle/" + path, ParticleAsset, cb)
	}

	/**
	 * 预加载所有资源 注意这里也是会影响其他资源的加载(占用网速)
	 */
	public static preloadAsset() {
		resources.preload('prefab/helpNode', Prefab)
		resources.preload('prefab/audioSettingNode', Prefab)
		resources.preload('prefab/autoShootNode', Prefab)
		resources.preload('prefab/netNode', Prefab)

		resources.preloadDir("sound", AudioClip)

		resources.preloadDir("particle", ParticleAsset, (err, data) => {
			if (!err) {
			}
		})

		resources.preloadDir("prefab/emitter", Prefab)
	}

	/** 
	* 预加载某一个资源
	*/
	public static preloadOneAsset<T extends Asset>(path: string, assetType: AssetType<T>, onComplete: (error: Error | undefined, data: object[]) => void) {
		resources.preload(path, assetType, onComplete)
	}

}

