/*
 * @Author: gongqinying (eycodes@gmail.com)
 * @Date: 2025-08-18 14:13:45
 * @LastEditTime: 2025-08-29 14:19:05
 * @LastEditors: gongqinying (eycodes@gmail.com)
 * @FilePath: \dev_client\assets\Script\data\ZipBundleLoader.ts
 * @Description: web平台解压加载资源
 */

import { SingtonClass } from "./SingtonClass";

declare const JSZip: any;

// 资源自动释放设置
const AUTO_RELEASE_INTERVAL = 5 * 60 * 1000; // 每5分钟检查
const AUTO_RELEASE_IDLE_TIME = 10 * 60 * 1000; // 空闲10分钟释放

class ZipBundleLoader extends SingtonClass {
    // 全局缓存：bundleName → JSZip对象
    // @ts-ignore
    bundleZipMap: Record<string, JSZip> = {};
    // 全局缓存：bundleName → 已解压的文件缓存（资源名 → Uint8Array）
    bundleFileMaps: Record<string, Record<string, { data: Uint8Array; lastAccess: number }>> = {};

    // 默认下载器引用
    defaultHandlers: Record<string, Function> = {
        // @ts-ignore
        '.json': cc.assetManager.downloader._downloaders['.json'],
        // @ts-ignore
        '.png': cc.assetManager.downloader._downloaders['.png'],
        // @ts-ignore
        '.plist': cc.assetManager.downloader._downloaders['.plist'],
        // @ts-ignore
        '.mp3': cc.assetManager.downloader._downloaders['.mp3'],
        // @ts-ignore
        '.ogg': cc.assetManager.downloader._downloaders['.ogg'],
        // @ts-ignore
        '.wav': cc.assetManager.downloader._downloaders['.wav'],
    };

    // 工具函数：从缓存或zip里取文件
    async getFileData(bundleName: string, relative: string): Promise<Uint8Array | null> {
        // 优先从已解压缓存取
        let fileCache = this.bundleFileMaps[bundleName]?.[relative];
        // if (fileData) return fileData;

        if (fileCache && fileCache.data) {
            // 更新最后访问时间
            fileCache.lastAccess = Date.now();
            return fileCache.data;
        }

        const zip = this.bundleZipMap[bundleName];
        if (zip) {
            const zipFile = zip.file(relative);
            if (zipFile) {
                const fileData = new Uint8Array(await zipFile.async('uint8array'));
                if (!this.bundleFileMaps[bundleName]) this.bundleFileMaps[bundleName] = {};
                this.bundleFileMaps[bundleName][relative] = { data: fileData, lastAccess: Date.now() };
                return fileData;
            }
        }
        return null;

        // // 从 zip 中单独解压
        // const zip = this.bundleZipMap[bundleName];
        // if (zip) {
        //     const zipFile = zip.file(relative);
        //     if (zipFile) {
        //         fileCache  = new Uint8Array(await zipFile.async('uint8array'));

        //         // 建立缓存
        //         if (!this.bundleFileMaps[bundleName]) this.bundleFileMaps[bundleName] = {};
        //         this.bundleFileMaps[bundleName][relative] = fileCache ;
        //         return fileCache ;
        //     }
        // }

        // return null;
    }

    // 注册全局下载器（只执行一次）
    registerGlobalZipLoaders() {
        // JSON
        cc.assetManager.downloader.register('.json', async (url, options, onComplete) => {
            const match = url.match(/assets\/([^/]+)\/(.+)/);
            if (match) {
                const [_, bundleName, relative] = match;
                const fileData = await this.getFileData(bundleName, relative);
                if (fileData) {
                    let text = new TextDecoder().decode(fileData);
                    if (text.charCodeAt(0) === 0xFEFF) text = text.slice(1); // 去BOM
                    try {
                        return onComplete(null!, JSON.parse(text));
                    } catch {
                        console.warn(`JSON parse failed: ${relative}`);
                        return onComplete(null!, text);
                    }
                }
            };

            this.defaultHandlers['.json'](url, options, onComplete);
        });

        // PNG
        cc.assetManager.downloader.register('.png', async (url, options, onComplete) => {
            const match = url.match(/assets\/([^/]+)\/(.+)/);
            if (match) {
                const [_, bundleName, relative] = match;
                const fileData = await this.getFileData(bundleName, relative);
                if (fileData) {
                    const blob = new Blob([fileData], { type: 'image/png' });
                    const img = new Image();
                    img.crossOrigin = 'anonymous';
                    img.onload = () => onComplete(null!, img);
                    img.onerror = () => onComplete(new Error(`Failed to load image: ${relative}`), null);
                    img.src = URL.createObjectURL(blob);
                    return;
                }
            }
            this.defaultHandlers['.png'](url, options, onComplete);
        });

        // Plist
        cc.assetManager.downloader.register('.plist', async (url, options, onComplete) => {
            const match = url.match(/assets\/([^/]+)\/(.+)/);
            if (match) {
                const [_, bundleName, relative] = match;
                const fileData = await this.getFileData(bundleName, relative);
                if (fileData) {
                    return onComplete(null!, new TextDecoder().decode(fileData));
                }
            }
            this.defaultHandlers['.plist'](url, options, onComplete);
        });

        // 音频
        const audioExts = ['.mp3', '.ogg', '.wav'];
        audioExts.forEach(ext => {
            cc.assetManager.downloader.register(ext, async (url, options, onComplete) => {
                const match = url.match(/assets\/([^/]+)\/(.+)/);
                if (match) {
                    const [_, bundleName, relative] = match;
                    const fileData = await this.getFileData(bundleName, relative);
                    if (fileData) {
                        return onComplete(null!, fileData.buffer); // ArrayBuffer
                    }
                }
                this.defaultHandlers[ext](url, options, onComplete);
            });
        });

        // 启动自动释放定时器
        setInterval(() => {
            const now = Date.now();
            for (const bundle in this.bundleFileMaps) {
                for (const key in this.bundleFileMaps[bundle]) {
                    const fileCache = this.bundleFileMaps[bundle][key];
                    if (fileCache && fileCache.data && now - fileCache.lastAccess > AUTO_RELEASE_IDLE_TIME) {
                        console.log(`[ZipLoader] Auto release file: ${bundle}/${key}`);
                        fileCache.data = null as any; // 释放引用
                    }
                }
            }
        }, AUTO_RELEASE_INTERVAL);
    }

    // 从远程加载并注册 zip bundle
    async loadZipBundle(bundleName: string) {
        // 如果已解压缓存存在，直接跳过下载和解压
        if (this.bundleZipMap[bundleName]) {
            console.log(`[ZipLoader] bundle '${bundleName}' 已在内存中，跳过解压`);
            return new Promise<void>((resolve, reject) => {
                // 如果 bundle 未注册到 assetManager，则重新 loadBundle
                if (!cc.assetManager.getBundle(bundleName)) {
                    cc.assetManager.loadBundle(bundleName, err => {
                        if (err) return reject(err);
                        resolve();
                    });
                } else {
                    resolve();
                }
            });
        }

        // 没有缓存则正常下载和解压
        // @ts-ignore
        const bundleVers = cc.assetManager.downloader.bundleVers;
        if (!bundleVers) return;

        const md5key: string = bundleVers[bundleName];
        const zipUrl = `assets/${bundleName}/${bundleName}.${md5key}.zip`; // 主包根目录下
        console.log(`[ZipLoader] 正在下载并解压 bundle '${bundleName}'...`);

        const response = await fetch(zipUrl);
        const arrayBuffer = await response.arrayBuffer();
        const zip = await JSZip.loadAsync(arrayBuffer);

        this.bundleZipMap[bundleName] = zip;
        this.bundleFileMaps[bundleName] = {}; // 初始化缓存

        return new Promise<void>((resolve, reject) => {
            cc.assetManager.loadBundle(bundleName, err => {
                if (err) {
                    delete this.bundleZipMap[bundleName];
                    delete this.bundleFileMaps[bundleName];
                    return reject(err);
                }
                resolve();
            });
        });







        // // @ts-ignore
        // const bundleVers = cc.assetManager.downloader.bundleVers;
        // if (!bundleVers) return;

        // const md5key: string = bundleVers[bundleName];
        // const zipUrl = `assets/${bundleName}/${bundleName}.${md5key}.zip`; // 主包根目录下
        // console.log(`加载本地 zip: ${zipUrl}`);

        // const response = await fetch(zipUrl);
        // const arrayBuffer = await response.arrayBuffer();
        // const zip = await JSZip.loadAsync(arrayBuffer);

        // this.bundleZipMap[bundleName] = zip;
        // this.bundleFileMaps[bundleName] = {}; // 初始化缓存

        // return new Promise<void>((resolve, reject) => {
        //     cc.assetManager.loadBundle(bundleName, err => {
        //         if (err) {
        //             delete this.bundleZipMap[bundleName];
        //             delete this.bundleFileMaps[bundleName];
        //             return reject(err);
        //         }
        //         resolve();
        //     });
        // });
    }

    // 释放 bundle 缓存 & zip 数据
    unloadZipBundle(bundleName: string) {
        const bundle = cc.assetManager.getBundle(bundleName);
        if (bundle) {
            bundle.releaseAll();
            cc.assetManager.removeBundle(bundle);
        }
        delete this.bundleZipMap[bundleName];
        delete this.bundleFileMaps[bundleName];
    }


















    // /** */
    // public static fileMap: Map<string, Uint8Array> = new Map();

    // /** */
    // public static unzipList: string[] = [];

    // public static register() {
    //     // @ts-ignore
    //     const defaultJsonHandler = cc.assetManager.downloader._downloaders['.json'];

    //     // @ts-ignore
    //     const defaultImageHandler = cc.assetManager.downloader._downloaders['.png'];

    //     // @ts-ignore
    //     const defaultPlistHandler = cc.assetManager.downloader._downloaders['.plist'];

    //     // @ts-ignore
    //     const defaultAudioHandler = cc.assetManager.downloader._downloaders['.mp3'] ||
    //         // @ts-ignore
    //         cc.assetManager.downloader._downloaders['.ogg'] ||
    //         // @ts-ignore
    //         cc.assetManager.downloader._downloaders['.wav'];

    //     // 3. 注册自定义下载器，拦截该分包路径
    //     cc.assetManager.downloader.register('.json', (url, options, onComplete) => {
    //         if (ZipBundleLoader.unzipList.find((bundleName) => { return url.includes(`assets/${bundleName}`) !== undefined })) {
    //             console.log("json==========>", url);
    //             const fileData = ZipBundleLoader.fileMap[url];
    //             if (!fileData) {
    //                 return onComplete(new Error('File not found in memory: ' + url), null);
    //             }

    //             // 假设是文本型
    //             let text = new TextDecoder().decode(fileData);
    //             // delete ZipBundleLoader.fileMap[url];
    //             return onComplete(null!, JSON.parse(text));
    //         };

    //         // if (url.includes(`assets/${bundleName}`)) {
    //         //     const fileData = ZipBundleLoader.fileMap[url];
    //         //     if (!fileData) {
    //         //         return onComplete(new Error('File not found in memory: ' + url), null);
    //         //     }

    //         //     // 假设是文本型
    //         //     let text = new TextDecoder().decode(fileData);
    //         //     delete ZipBundleLoader.fileMap[url];
    //         //     return onComplete(null!, JSON.parse(text));
    //         // }

    //         // 交给原始处理器
    //         defaultJsonHandler(url, options, onComplete);
    //     });

    //     cc.assetManager.downloader.register('.png', (url, options, onComplete) => {
    //         if (ZipBundleLoader.unzipList.find((bundleName) => { return url.includes(`assets/${bundleName}`) !== undefined })) {
    //             const fileData = ZipBundleLoader.fileMap[url];
    //             if (!fileData) {
    //                 return onComplete(new Error('File not found in memory: ' + url), null);
    //             }
    //             const blob = new Blob([fileData], { type: "image/png" });
    //             const img = new Image();
    //             img.onload = () => {
    //                 URL.revokeObjectURL(img.src);
    //                 // delete ZipBundleLoader.fileMap[url];
    //                 onComplete(null!, img);
    //             };

    //             img.onerror = () => {
    //                 URL.revokeObjectURL(img.src);
    //                 // delete ZipBundleLoader.fileMap[url];
    //                 onComplete(new Error('Failed to load image: ' + url), null);
    //             };
    //             img.src = URL.createObjectURL(blob);
    //             return;
    //         }



    //         // if (url.includes(`assets/${bundleName}`)) {
    //         //     const fileData = ZipBundleLoader.fileMap[url];
    //         //     if (!fileData) {
    //         //         return onComplete(new Error('File not found in memory: ' + url), null);
    //         //     }
    //         //     const blob = new Blob([fileData], { type: "image/png" });
    //         //     const img = new Image();
    //         //     img.onload = () => {
    //         //         URL.revokeObjectURL(img.src);
    //         //         delete ZipBundleLoader.fileMap[url];
    //         //         onComplete(null!, img);
    //         //     };

    //         //     img.onerror = () => {
    //         //         URL.revokeObjectURL(img.src);
    //         //         delete ZipBundleLoader.fileMap[url];
    //         //         onComplete(new Error('Failed to load image: ' + url), null);
    //         //     };
    //         //     img.src = URL.createObjectURL(blob);
    //         //     return;
    //         // }

    //         // 交给原始处理器
    //         defaultImageHandler(url, options, onComplete);
    //     });

    //     // ---------------- PLIST ----------------
    //     cc.assetManager.downloader.register('.plist', (url, options, onComplete) => {
    //         if (ZipBundleLoader.unzipList.find((bundleName) => { return url.includes(`assets/${bundleName}`) !== undefined })) {
    //             const fileData = ZipBundleLoader.fileMap[url];
    //             if (!fileData) return onComplete(new Error('File not found: ' + url), null);
    //             const text = new TextDecoder().decode(fileData);
    //             // delete ZipBundleLoader.fileMap[url];
    //             return onComplete(null!, text); // Cocos 内部会解析
    //         }



    //         // if (url.includes(`assets/${bundleName}`)) {
    //         //     const fileData = fileMap[url];
    //         //     if (!fileData) return onComplete(new Error('File not found: ' + url), null);
    //         //     const text = new TextDecoder().decode(fileData);
    //         //     delete ZipBundleLoader.fileMap[url];
    //         //     return onComplete(null!, text); // Cocos 内部会解析
    //         // }
    //         defaultPlistHandler(url, options, onComplete);
    //     });

    //     // ---------------- Audio ----------------
    //     const audioExts = ['.mp3', '.ogg', '.wav'];
    //     audioExts.forEach(ext => {
    //         cc.assetManager.downloader.register(ext, (url, options, onComplete) => {
    //             if (ZipBundleLoader.unzipList.find((bundleName) => { return url.includes(`assets/${bundleName}`) !== undefined })) {
    //                 const fileData = ZipBundleLoader.fileMap[url];
    //                 if (!fileData) return onComplete(new Error('File not found: ' + url), null);

    //                 let mimeType = '';
    //                 if (ext === '.mp3') mimeType = 'audio/mpeg';
    //                 else if (ext === '.ogg') mimeType = 'audio/ogg';
    //                 else if (ext === '.wav') mimeType = 'audio/wav';

    //                 // delete ZipBundleLoader.fileMap[url];
    //                 return onComplete(null!, fileData.buffer); // Cocos 会接管 blob 播放
    //             }


    //             // if (url.includes(`assets/${bundleName}`)) {
    //             //     const fileData = ZipBundleLoader.fileMap[url];
    //             //     if (!fileData) return onComplete(new Error('File not found: ' + url), null);

    //             //     let mimeType = '';
    //             //     if (ext === '.mp3') mimeType = 'audio/mpeg';
    //             //     else if (ext === '.ogg') mimeType = 'audio/ogg';
    //             //     else if (ext === '.wav') mimeType = 'audio/wav';

    //             //     delete ZipBundleLoader.fileMap[url];
    //             //     return onComplete(null!, fileData.buffer); // Cocos 会接管 blob 播放
    //             // }
    //             defaultAudioHandler(url, options, onComplete);
    //         });
    //     });
    // }

    // public static async load(bundleName: string) {
    //     // // @ts-ignore
    //     // const defaultJsonHandler = cc.assetManager.downloader._downloaders['.json'];

    //     // // @ts-ignore
    //     // const defaultImageHandler = cc.assetManager.downloader._downloaders['.png'];

    //     // // @ts-ignore
    //     // const defaultPlistHandler = cc.assetManager.downloader._downloaders['.plist'];

    //     // // @ts-ignore
    //     // const defaultAudioHandler = cc.assetManager.downloader._downloaders['.mp3'] ||
    //     //     // @ts-ignore
    //     //     cc.assetManager.downloader._downloaders['.ogg'] ||
    //     //     // @ts-ignore
    //     //     cc.assetManager.downloader._downloaders['.wav'];

    //     // @ts-ignore
    //     const bundleVers = cc.assetManager.downloader.bundleVers;
    //     if (!bundleVers) return;

    //     const md5key: string = bundleVers[bundleName];
    //     const zipUrl = `assets/${bundleName}/${bundleName}.${md5key}.zip`; // 主包根目录下
    //     console.log(`加载本地 zip: ${zipUrl}`);

    //     if (!ZipBundleLoader.unzipList.includes(bundleName)) {
    //         // 1. 下载 zip
    //         let jszip = new JSZip();
    //         const res = await fetch(zipUrl);
    //         if (!res.ok) throw new Error(`加载 zip 失败: ${res.statusText}`);
    //         const zipData = await res.arrayBuffer();
    //         const zip = await jszip.loadAsync(zipData);

    //         // 2. 存储文件数据到一个 map
    //         // const fileMap: Record<string, Uint8Array> = {};
    //         for (const key of Object.keys(zip.files)) {
    //             const file = zip.files[key];
    //             if (!file.dir) {
    //                 ZipBundleLoader.fileMap[`assets/${bundleName}/${key}`] = await file.async('uint8array');
    //             };
    //         };

    //         ZipBundleLoader.unzipList.push(bundleName);
    //         // console.log("ZipBundleLoader.fileMap====>", ZipBundleLoader.fileMap);
    //     };












    // // 3. 注册自定义下载器，拦截该分包路径
    // cc.assetManager.downloader.register('.json', (url, options, onComplete) => {
    //     if (url.includes(`assets/${bundleName}`)) {
    //         const fileData = fileMap[url];
    //         if (!fileData) {
    //             return onComplete(new Error('File not found in memory: ' + url), null);
    //         }

    //         // 假设是文本型
    //         let text = new TextDecoder().decode(fileData);
    //         delete fileMap[url];
    //         return onComplete(null!, JSON.parse(text));
    //     }

    //     // 交给原始处理器
    //     defaultJsonHandler(url, options, onComplete);
    // });

    // cc.assetManager.downloader.register('.png', (url, options, onComplete) => {
    //     if (url.includes(`assets/${bundleName}`)) {
    //         const fileData = fileMap[url];
    //         if (!fileData) {
    //             return onComplete(new Error('File not found in memory: ' + url), null);
    //         }
    //         const blob = new Blob([fileData], { type: "image/png" });
    //         const img = new Image();
    //         img.onload = () => {
    //             URL.revokeObjectURL(img.src);
    //             delete fileMap[url];
    //             onComplete(null!, img);
    //         };

    //         img.onerror = () => {
    //             URL.revokeObjectURL(img.src);
    //             delete fileMap[url];
    //             onComplete(new Error('Failed to load image: ' + url), null);
    //         };
    //         img.src = URL.createObjectURL(blob);
    //         return;
    //     }

    //     // 交给原始处理器
    //     defaultImageHandler(url, options, onComplete);
    // });

    // // ---------------- PLIST ----------------
    // cc.assetManager.downloader.register('.plist', (url, options, onComplete) => {
    //     if (url.includes(`assets/${bundleName}`)) {
    //         const fileData = fileMap[url];
    //         if (!fileData) return onComplete(new Error('File not found: ' + url), null);
    //         const text = new TextDecoder().decode(fileData);
    //         delete fileMap[url];
    //         return onComplete(null!, text); // Cocos 内部会解析
    //     }
    //     defaultPlistHandler(url, options, onComplete);
    // });

    // // ---------------- Audio ----------------
    // const audioExts = ['.mp3', '.ogg', '.wav'];
    // audioExts.forEach(ext => {
    //     cc.assetManager.downloader.register(ext, (url, options, onComplete) => {
    //         if (url.includes(`assets/${bundleName}`)) {
    //             const fileData = fileMap[url];
    //             if (!fileData) return onComplete(new Error('File not found: ' + url), null);

    //             let mimeType = '';
    //             if (ext === '.mp3') mimeType = 'audio/mpeg';
    //             else if (ext === '.ogg') mimeType = 'audio/ogg';
    //             else if (ext === '.wav') mimeType = 'audio/wav';

    //             delete fileMap[url];
    //             return onComplete(null!, fileData.buffer); // Cocos 会接管 blob 播放
    //         }
    //         defaultAudioHandler(url, options, onComplete);
    //     });
    // });

    // // 4. 加载分包（会走我们注册的 handler）
    // cc.assetManager.loadBundle(bundleName, (err, bundle) => {
    //     if (err) {
    //         console.error(err);
    //     } else {
    //         console.log(`${bundleName} loaded from memory`);
    //     }
    // });
}
// }

export const ZipLoader = ZipBundleLoader.getInstance();