/*
 * Copyright (c) 2010 Nathan Rajlich (https://github.com/TooTallNate)
 * Copyright (c) 2010 Animesh Kumar (https://github.com/anismiles)
 * Copyright (c) 2010 Strumsoft (https://strumsoft.com)
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */
package com.strumsoft.websocket.phonegap;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.InetSocketAddress;
import java.net.URI;
import java.nio.ByteBuffer;
import java.nio.channels.NotYetConnectedException;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.SocketChannel;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Iterator;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

import android.app.Activity;
import android.content.Context;
import android.os.Handler;
import android.os.Message;
import android.util.Log;
import android.view.inputmethod.InputMethodManager;
import android.webkit.JavascriptInterface;
import android.webkit.WebView;

/**
 * The <tt>WebSocket</tt> is an implementation of WebSocket Client API, and
 * expects a valid "ws://" URI to connect to. When connected, an instance
 * recieves important events related to the life of the connection, like
 * <var>onOpen</var>, <var>onClose</var>, <var>onError</var> and
 * <var>onMessage</var>. An instance can send messages to the server via the
 * <var>send</var> method.
 *
 * @author Animesh Kumar
 */
public class WebSocket implements Runnable {

	/**
	 * Enum for WebSocket Draft
	 */
	public enum Draft {
		DRAFT75, DRAFT76
	}

	// //////////////// CONSTANT
	/**
	 * The connection has not yet been established.
	 */
	public final static int WEBSOCKET_STATE_CONNECTING = 0;
	/**
	 * The WebSocket connection is established and communication is possible.
	 */
	public final static int WEBSOCKET_STATE_OPEN = 1;
	/**
	 * The connection is going through the closing handshake.
	 */
	public final static int WEBSOCKET_STATE_CLOSING = 2;
	/**
	 * The connection has been closed or could not be opened.
	 */
	public final static int WEBSOCKET_STATE_CLOSED = 3;

	/**
	 * An empty string
	 */
	private static String BLANK_MESSAGE = "";
	/**
	 * The javascript method name for onOpen event.
	 */
	private static String EVENT_ON_OPEN = "onopen";
	/**
	 * The javascript method name for onMessage event.
	 */
	private static String EVENT_ON_MESSAGE = "onmessage";
	/**
	 * The javascript method name for onClose event.
	 */
	private static String EVENT_ON_CLOSE = "onclose";
	/**
	 * The javascript method name for onError event.
	 */
	private static String EVENT_ON_ERROR = "onerror";
	/**
	 * The default port of WebSockets, as defined in the spec.
	 */
	public static final int DEFAULT_PORT = 80;
	/**
	 * The WebSocket protocol expects UTF-8 encoded bytes.
	 */
	public static final String UTF8_CHARSET = "UTF-8";
	/**
	 * The byte representing Carriage Return, or \r
	 */
	public static final byte DATA_CR = (byte) 0x0D;
	/**
	 * The byte representing Line Feed, or \n
	 */
	public static final byte DATA_LF = (byte) 0x0A;
	/**
	 * The byte representing the beginning of a WebSocket text frame.
	 */
	public static final byte DATA_START_OF_FRAME = (byte) 0x00;
	/**
	 * The byte representing the end of a WebSocket text frame.
	 */
	public static final byte DATA_END_OF_FRAME = (byte) 0xFF;

	// //////////////// INSTANCE Variables
	/**
	 * The WebView instance from Phonegap DroidGap
	 */
	private final WebView appView;
	/**
	 * The unique id for this instance (helps to bind this to javascript events)
	 */
	private String id;
	/**
	 * The URI this client is supposed to connect to.
	 */
	private URI uri;
	/**
	 * The port of the websocket server
	 */
	private int port;
	/**
	 * The Draft of the WebSocket protocol the Client is adhering to.
	 */
	private Draft draft;
	/**
	 * The <tt>SocketChannel</tt> instance to use for this server connection.
	 * This is used to read and write data to.
	 */
	private SocketChannel socketChannel;
	/**
	 * The 'Selector' used to get event keys from the underlying socket.
	 */
	private Selector selector;
	/**
	 * Keeps track of whether or not the client thread should continue running.
	 */
	private boolean running;
	/**
	 * Internally used to determine whether to recieve data as part of the
	 * remote handshake, or as part of a text frame.
	 */
	private boolean handshakeComplete = true;
	/**
	 * The 1-byte buffer reused throughout the WebSocket connection to read
	 * data.
	 */
	private ByteBuffer buffer;
	/**
	 * The bytes that make up the remote handshake.
	 */
	private ByteBuffer remoteHandshake;
	/**
	 * The bytes that make up the current text frame being read.
	 */
	private ByteBuffer currentFrame;
	/**
	 * Queue of buffers that need to be sent to the client.
	 */
	private BlockingQueue<ByteBuffer> bufferQueue;
	/**
	 * Lock object to ensure that data is sent from the bufferQueue in the
	 * proper order
	 */
	private Object bufferQueueMutex = new Object();
	/**
	 * Number 1 used in handshake
	 */
	private int number1 = 0;
	/**
	 * Number 2 used in handshake
	 */
	private int number2 = 0;
	/**
	 * Key3 used in handshake
	 */
	private byte[] key3 = null;
	/**
	 * The readyState attribute represents the state of the connection.
	 */
	private int readyState = WEBSOCKET_STATE_CONNECTING;

	private boolean keyboardIsShowing = false;

	private Handler handler = null;

	private final WebSocket instance;

	private ByteBuffer bigBuffer = ByteBuffer.allocate(1024 * 500);
	private byte[] tokenByteBuffer = new byte[1024 * 500];
	private int tokenByteBufferCounter = 0;

	/**
	 * Constructor.
	 *
	 * Note: this is protected because it's supposed to be instantiated from {@link WebSocketFactory} only.
	 *
	 * @param appView
	 *            {@link android.webkit.WebView}
	 * @param uri
	 *            websocket server {@link URI}
	 * @param draft
	 *            websocket server {@link Draft} implementation (75/76)
	 * @param id
	 *            unique id for this instance
	 */
	protected WebSocket(Handler handler, WebView appView, URI uri, Draft draft, String id) {
		this.appView = appView;
		this.uri = uri;
		this.draft = draft;
		this.handler = handler;

		// port
		port = uri.getPort();
		if (port == -1) {
			port = DEFAULT_PORT;
		}

		// Id
		this.id = id;

		this.bufferQueue = new LinkedBlockingQueue<ByteBuffer>();
		this.handshakeComplete = false;
		this.remoteHandshake = this.currentFrame = null;
		this.buffer = ByteBuffer.allocate(1);

		this.instance = this;
	}

	// //////////////////////////////////////////////////////////////////////////////////////
	// /////////////////////////// WEB SOCKET API Methods
	// ///////////////////////////////////
	// //////////////////////////////////////////////////////////////////////////////////////
	/**
	 * Starts a new Thread and connects to server
	 *
	 * @throws IOException
	 */
	public Thread connect() throws IOException {
		this.running = true;
		this.readyState = WEBSOCKET_STATE_CONNECTING;
		// open socket
		socketChannel = SocketChannel.open();
		socketChannel.configureBlocking(false);
		// set address
		socketChannel.connect(new InetSocketAddress(uri.getHost(), port));
		// start a thread to make connection
		Log.d("websocket", "host: "+uri.getHost() + " port:" + port);


		// More info:
		// http://groups.google.com/group/android-developers/browse_thread/thread/45a8b53e9bf60d82
		// http://stackoverflow.com/questions/2879455/android-2-2-and-bad-address-family-on-socket-connect
		System.setProperty("java.net.preferIPv4Stack", "true");
		//System.setProperty("java.net.preferIPv6Addresses", "false");

		selector = Selector.open();
		socketChannel.register(selector, SelectionKey.OP_CONNECT);
		Log.v("websocket", "Starting a new thread to manage data reading/writing");

		Thread th = new Thread(this);
		th.start();
		// return thread object for explicit closing, if needed
		return th;
	}

	public void run() {
		while (this.running) {
			try {
				_connect();
			} catch (IOException e) {
				e.printStackTrace();
				this.onError(e);
			}
		}
	}


	public void setKeyboardStatus(boolean status){
		keyboardIsShowing = status;
		Log.d("websocket", "keyboardIsShowing: "+keyboardIsShowing);
	}
	/**
	 * Closes connection with server
	 */
	@JavascriptInterface
	public void close() {
		this.readyState = WebSocket.WEBSOCKET_STATE_CLOSING;

		// close socket channel
		try {
			this.socketChannel.close();
		} catch (IOException e) {
			this.onError(e);
		}
		this.running = false;
		selector.wakeup();

		// fire onClose method
		this.onClose();

		this.readyState = WebSocket.WEBSOCKET_STATE_CLOSED;
	}

	/**
	 * Sends <var>text</var> to server
	 *
	 * @param text
	 *            String to send to server
	 */
	@JavascriptInterface
	public void send(byte[] text) {
		new Thread(new Runnable() {
			@Override
			public void run() {
				if (instance.readyState == WEBSOCKET_STATE_OPEN) {
					try {
						instance._send(text);
					} catch (Exception e) {
						instance.onError(e);
					}
				} else {
					instance.onError(new NotYetConnectedException());
				}
			}
		}).start();
	}

	/**
	 * Called when an entire text frame has been received.
	 *
	 * @param bytes
	 *            Message from websocket server
	 */
	public void onMessage(byte[] bytes) {
		final byte[] data = bytes;
		appView.post(new Runnable() {
	        public void run() {
	            if(keyboardIsShowing){
	            	Message message = new Message();
	            	message.obj = buildJavaScriptData(EVENT_ON_MESSAGE, data);
	            	message.what = 3;
	            	handler.sendMessage(message);
	            } else {
					Log.v("websocket", "onMessage len " + data.length );
	            	appView.loadUrl(buildJavaScriptData(EVENT_ON_MESSAGE, data));
	            }
	        }
	    });
	}

	public void onOpen() {
		Log.v("websocket", "Connected!");
		appView.post(new Runnable() {
	        public void run() {
	            appView.loadUrl(buildJavaScriptData(EVENT_ON_OPEN,BLANK_MESSAGE.getBytes()));
	            if(keyboardIsShowing){
	            	handler.sendEmptyMessage(3);
	            }
	        }
	    });
	}

	public void onClose() {
		appView.post(new Runnable() {
	      public void run() {
	          appView.loadUrl(buildJavaScriptData(EVENT_ON_CLOSE, BLANK_MESSAGE.getBytes()));
	          if(keyboardIsShowing){
	          	handler.sendEmptyMessage(3);
	          }
	      }
    });
	}

	public void onError(Throwable t) {
		final String msg = t.getMessage();
		Log.v("websocket", "Error: " + msg);
		t.printStackTrace();
		appView.post(new Runnable() {
	        public void run() {
	            appView.loadUrl(buildJavaScriptData(EVENT_ON_ERROR, msg.getBytes()));
	            if(keyboardIsShowing){
	            	handler.sendEmptyMessage(3);
	            }
	        }
	    });
	}

	@JavascriptInterface
	public String getId() {
		return id;
	}

	/**
	 * @return the readyState
	 */
	@JavascriptInterface
	public int getReadyState() {
		return readyState;
	}

	/**
	 * Builds text for javascript engine to invoke proper event method with
	 * proper data.
	 *
	 * @param event
	 *            websocket event (onOpen, onMessage etc.)
	 * @param bytes
	 *            Text message received from websocket server
	 * @return
	 */
	private String buildJavaScriptData(String event, byte[] bytes) {
		String b64EncodedMsg = "Error!";
		try{
			if(bytes != null) {
				b64EncodedMsg = com.strumsoft.phonegap.websocket.Base64.encodeBytes(bytes);
			}
		} catch(Exception e) {
			e.printStackTrace();
		}

		String _d = "javascript:WebSocket." + event + "(" + "{" + "\"_target\":\"" + id + "\","
				+ "\"_data\":'" + b64EncodedMsg + "'" + "}" + ")";
		return _d;
	}

	// //////////////////////////////////////////////////////////////////////////////////////
	// /////////////////////////// WEB SOCKET Internal Methods
	// //////////////////////////////
	// //////////////////////////////////////////////////////////////////////////////////////

	private boolean _send(byte[] bytes) throws IOException {
		if (bytes == null) {
			throw new NullPointerException("Cannot send 'null' data to a WebSocket.");
		}

		ByteBuffer b = ByteBuffer.wrap(bytes);

		if (_write()) {
			Log.d("websocket ","_write : ");
			this.socketChannel.write(b);
		}

		// If we didn't get it all sent, add it to the buffer of buffers
		if (b.remaining() > 0) {
			if (!this.bufferQueue.offer(b)) {
				throw new IOException("Buffers are full, message could not be sent to"
						+ this.socketChannel.socket().getRemoteSocketAddress());
			}
			return false;
		}
		return true;
	}

	// actual connection logic
	private void _connect() throws IOException {
		// Continuous loop that is only supposed to end when "close" is called.

		selector.select();
		Set<SelectionKey> keys = selector.selectedKeys();
		Iterator<SelectionKey> i = keys.iterator();

		while (i.hasNext()) {
			SelectionKey key = i.next();
			i.remove();
			if (key.isConnectable()) {
				if (socketChannel.isConnectionPending()) {
					socketChannel.finishConnect();
				}
				socketChannel.register(selector, SelectionKey.OP_READ);

				this.readyState = WEBSOCKET_STATE_OPEN;
				// fire onOpen method
				this.onOpen();
			}
			if (key.isReadable()) {
				try {
					_read();
				} catch (NoSuchAlgorithmException nsa) {
					this.onError(nsa);
				}
			}
		}

	}

	private boolean _write() throws IOException {
		synchronized (this.bufferQueueMutex) {
			ByteBuffer buffer = this.bufferQueue.peek();
			while (buffer != null) {
				this.socketChannel.write(buffer);
				if (buffer.remaining() > 0) {
					return false; // Didn't finish this buffer. There's more to
				} else {
					this.bufferQueue.poll(); // Buffer finished. Remove it.
					buffer = this.bufferQueue.peek();
				}
			}
			return true;
		}
	}

	private void _write(byte[] bytes) throws IOException {
		this.socketChannel.write(ByteBuffer.wrap(bytes));
	}

	private void _read() throws IOException, NoSuchAlgorithmException {
		int bytesRead = -1;
		try {
				bigBuffer.rewind();
				bytesRead = socketChannel.read(this.bigBuffer);
				bigBuffer.rewind();

		} catch (Exception ex) {
			Log.v("websocket", "Could not read data from socket channel, ex=" + ex.toString());
		}


		if (bytesRead == -1) {
			Log.v("websocket", "All Bytes readed");
			close();
		} else if (bytesRead > 0) {
			_readFrame(bytesRead);
		}
	}

	private void _readFrame(int bytesRead) throws UnsupportedEncodingException {


		byte[] data = bigBuffer.array();

		byte[] _bytes = new byte[bytesRead];

		for(int i=0;i<bytesRead;i++){
			_bytes[i] = data[i];
		}

		this.onMessage(_bytes);
	}
}
