import FileList from "./FileList.js";
import FileTool from "./FileTool.js";
import { I18ChangeTxtBundle } from "./I18ChangeTxtBundle.js";

class ResType {    
    static _mapResType = new Map();

    static initResType(){
        const self = this;
        self._mapResType.clear();
        self._mapResType.set( "cc.SpriteFrame", "sprite-frame" );
        self._mapResType.set( "sprite-frame", "cc.SpriteFrame" );
        
        self._mapResType.set( "bitmap-font", "cc.BitmapFont" );        
        self._mapResType.set( "cc.BitmapFont", "bitmap-font" );
    }

    static isSameResType( type1, type2 ){
        return (type1 == type2) || (ResType._mapResType.get( type1 ) == type2);
    }
}

class ResInfo{
    uuid = "";
    url = "";
    type = "";
    res = [];   //[ { uuid:"", type:"sprite-frame" } ]

    getResByUUID( uuid ){
        const self = this;
        for( const res of self.res ){
            if( res.uuid == uuid ){
                return res;
            }
        }
        return null;
    }

    getResByType( type ){
        const self = this;
        for( const res of self.res ){
            if( res.type == type ){
                return res;
            }
        }
        return null;
    }

    getUUIDByType( type ){
        const self = this;
        for( const res of self.res ){
            if( ResType.isSameResType( res.type, type ) ){
                return self.uuid + "@" + res.uuid;
            }
        }
        
        if( ResType.isSameResType( self.type, type ) ){
            return self.uuid;
        }
        
        return null;
    }

    addType( uuid, type ){
        const self = this;
        if( self.getResByUUID( uuid ) ) return;
        const res = { uuid:uuid, type:type };
        self.res.push( res );
    }

}

export class I18ChangeBundle {

    _assetDir = "";    
    _defaultDir = "";
    _newDir = "";

    _assetFile = [];
    _mapTxt = new Map();
    _mapDefaultRes = new Map();
    _mapNewRes = new Map();

    _arrModifyFile = [];
    _mapReplaceUUID = new Map();

    _mapNewName = new Map();

    constructor( assetDir, defaultDir, newDir ){
        const self = this;
        self._assetDir = assetDir;
        self._defaultDir = defaultDir;
        self._newDir = newDir;
        ResType.initResType();
        self.initNewName();
    }

    getDirName( dir ){
        const idx = dir.lastIndexOf( "/", dir.length - 2 );
        const idx2 = dir.lastIndexOf( "/" );
        return dir.substr( idx + 1, idx2 - idx - 1 );
    }

    getParentDir( dir ){
        const idx = dir.lastIndexOf( "/", dir.length - 2 );
        return dir.substr( 0, idx );
    }

    initNewName(){
        const self = this;
        const name1 = self.getDirName( self._defaultDir );
        const name2 = self.getDirName( self._newDir );
        self._mapNewName.set( name1, name2 );
        self._mapNewName.set( name2, name1 );
    }

    initAsset(){
        console.log( "--------init asset dir--------" );
        const self = this;
        const fl = new FileList();
        fl.initData( self._assetDir );
        self._assetFile = [];

        const allScene = fl.filterByType( "scene" );
        allScene.forEach((fileName)=>{
            self._assetFile.push( self._assetDir + fileName );
        });

        const allPrefab = fl.filterByType( "prefab" );
        allPrefab.forEach((fileName)=>{
            self._assetFile.push( self._assetDir + fileName );
        });
    }

    getUrlByFileName( fileName ){
        // const startIdx = fileName.lastIndexOf( "/" );
        // const endIdx = fileName.indexOf( ".", startIdx+1 );
        // return fileName.substr( startIdx+1, endIdx );
        
        const endIdx = fileName.indexOf( "." );
        return fileName.substr( 0, endIdx );
    }

    getResInfoInFile( dir, fileName ){
        const self = this;

        console.log( "read file:" + fileName );
        const jsonData = FileTool.readJsonInFile( dir + fileName );
        if( jsonData == null ){
            console.log( "read json failt !" );
            return null;
        }

        const uuid = jsonData[ "uuid" ];
        if( uuid == null || uuid == "" ){
            console.log( "read uuid failt !" );
            return null;
        }

        const subMetas = jsonData[ "subMetas" ];
        if( subMetas == null ){
            console.log( "read subMetas failt !" );
            return null;
        }

        const url = self.getUrlByFileName( fileName );

        const resInfo = new ResInfo();
        resInfo.uuid = uuid;
        resInfo.url = url;
        resInfo.type = jsonData[ "importer" ];
        for( const subInfoKey in subMetas ){
            const subInfo = subMetas[ subInfoKey ];
            resInfo.addType( subInfoKey, subInfo[ "importer" ] );
        }
        return resInfo;
    }

    getResByFullUUID( mapRes, fullUUID ){
        const idx = fullUUID.indexOf( "@" );
        let uuid = "";
        if( idx != -1 ){
            uuid = fullUUID.substr( 0, idx );
        }else{
            uuid = fullUUID;
        }

        return mapRes.get( uuid );
    }

    getResInfoByUrl( mapRes, url ){
        for( const [uuid, resInfo] of mapRes ){
            if( resInfo.url == url ){
                return resInfo;
            }
        }
        return null;
    }

    initRes( dir, mapRes ){
        console.log( "--------init res dir--------" );
        console.log( dir );

        const self = this;
        const fl = new FileList();
        fl.initData( dir );
        const arrFile = fl.filterByStr( ".png.meta" );
        arrFile.forEach((fileName)=>{
            const resInfo = self.getResInfoInFile( dir, fileName );
            mapRes.set( resInfo.uuid, resInfo );
        });
    }

    initLanguageDir(){
        const self = this;
        self.initRes( self._defaultDir, self._mapDefaultRes );
        self.initRes( self._newDir, self._mapNewRes );
    }

    checkJsonTypeHasUUID( jsonType ){
        if( jsonType == null ) return true;
        if( jsonType == "cc.Node" ) return false;
        if( jsonType == "cc.Prefab" ) return false;
        if( jsonType == "cc.Scene" ) return false;
        if( jsonType == "cc.SceneAsset" ) return false;
        if( jsonType == "cc.Camera" ) return false;
        if( jsonType == "cc.UIOpacity" ) return false;
        if( jsonType == "cc.UITransform" ) return false;
        if( jsonType == "cc.Vec3" ) return false;
        if( jsonType == "cc.Quat" ) return false;
        return true;
    }

    getUUIDByJsonData( jsonData, arrUUIDInfo ){
        if( jsonData == null ) return;

        const self = this;
        if( !self.checkJsonTypeHasUUID( jsonData[ "__type__" ] ) ) return;

        for( const k in jsonData ){
            if( k == "__uuid__" ){
                arrUUIDInfo.push( { uuid: jsonData[k], type: jsonData["__expectedType__"] } );
            }else if( typeof jsonData[k] == "object" ){
                self.getUUIDByJsonData( jsonData[k], arrUUIDInfo );
            }
        }
    }

    findReplaceUUID(){
        console.log( "--------find replace uuid--------" );
        const self = this;

        self._arrModifyFile = [];
        self._mapReplaceUUID.clear();

        let bError = false;
        for( const fullName of self._assetFile ){
            console.log( "handle file: " + fullName );
            const jsonData = FileTool.readJsonInFile( fullName );
            if( jsonData == null || !Array.isArray(jsonData) ){
                bError = true;
                console.log( "read json failt: " + fullName );
                break;
            }

            const arrUUIDInfo = [];
            jsonData.forEach( (jsData)=>{
                self.getUUIDByJsonData( jsData, arrUUIDInfo );
            } );

            let findCount = 0;
            for( const uuidInfo of arrUUIDInfo ){
                if( self._mapReplaceUUID.get( uuidInfo.uuid ) ){
                    ++findCount;
                    continue;
                }

                const defaultResInfo = self.getResByFullUUID( self._mapDefaultRes, uuidInfo.uuid );
                if( defaultResInfo == null ) continue;

                const url = defaultResInfo.url;
                const newResInfo = self.getResInfoByUrl( self._mapNewRes, url );
                if( newResInfo == null ){
                    bError = true;
                    console.error( `not found res url:${url}` );
                    break;
                }

                const newUUID = newResInfo.getUUIDByType( uuidInfo.type );
                if( newUUID == null ){
                    bError = true;
                    console.error( `not found res url:${url} type:${uuidInfo.type}` );
                    break;
                } 

                self._mapReplaceUUID.set( uuidInfo.uuid, newUUID );
                ++findCount;
            }

            if( bError ) break;

            if( findCount > 0 ){
                self.pushModifyFile( fullName );
            }
        }

        return !bError;
    }

    pushModifyFile( fileName ){
        const self = this;
        if( self._arrModifyFile.indexOf(fileName) != -1 ) return;
        self._arrModifyFile.push( fileName );
    }

    checkJsonByType( jsonData, strType ){
            if( jsonData == null ) return false;
            if( jsonData[ "__type__" ] == null ) return false;
            return jsonData[ "__type__" ] == strType;
    }

    checkSpriteType( jsonData ){
        return this.checkJsonByType( jsonData, "cc.Sprite" );
    }

    getNodeData( jsonData, id, type ){        
        if( jsonData == null ) return null;
        if( !Array.isArray(jsonData) ) return null;

        const self = this;
        const data = jsonData[ id ];
        if( data == null ) return null;

        if( !self.checkJsonByType( data, type ) ) return null;
        return data;
    }

    getUITransfromByID( jsonData, id ){
        return this.getNodeData( jsonData, id, "cc.UITransform" );
    }

    getUITransfromByNodeId( jsonData, nodeId ){
        if( jsonData == null ) return null;
        if( !Array.isArray(jsonData) ) return null;

        for( const cmpt of jsonData ){
            if( cmpt[ "__type__" ] != "cc.UITransform" ) continue;

            if( cmpt[ "node" ] == null ) continue;
            if( cmpt[ "node" ][ "__id__" ] == nodeId )
                return cmpt;
        }

        return null;
    }

    modifyUITransfromByID( jsonData, id, size ){
        const self = this;
        const uiTransform = self.getUITransfromByID( jsonData, id );
        if( uiTransform == null ) return;

        const cmptSize = uiTransform[ "_contentSize" ];
        if( cmptSize == null ) return;

        cmptSize.width = size.width;
        cmptSize.height = size.height;
    }

    modifySpriteFrameUUID( jsonData ){
        const self = this;
        if( !self.checkSpriteType(jsonData) ) return null;
     
        const spriteFrame = jsonData[ "_spriteFrame" ];
        if( spriteFrame == null ) return null;

        const uuid = spriteFrame[ "__uuid__" ];
        if( uuid == null ) return null;

        const newUUID = self._mapReplaceUUID.get( uuid );
        if( newUUID == null ) return null;

        spriteFrame[ "__uuid__" ] = newUUID;
        return newUUID;
    }

    getBaseUUID( uuid ){
        const idx = uuid.indexOf( "@" );
        if( idx == -1 ){
            return uuid;
        }else{
            return uuid.substr( 0, idx );
        }
    }

    modifySpriteInFileJson( jsonData ){
        const self = this;
        if( jsonData == null ) return;
        if( !Array.isArray(jsonData) ) return;

        for( const jsData of jsonData ){
            const uuid = self.modifySpriteFrameUUID( jsData );
            if( uuid == null ) continue;

            const resInfo = self._mapNewRes.get( self.getBaseUUID(uuid) );
            if( resInfo == null ) continue;

            const imgFile = self._newDir + resInfo.url + ".png";
            const size = FileTool.getImageSize( imgFile );
            if( size == null ) continue;

            if( jsData[ "node" ] == null ) continue;
            const nodeId = jsData[ "node" ][ "__id__" ];

            const uiTransform = self.getUITransfromByNodeId( jsonData, nodeId );
            if( uiTransform == null ) continue;

            const uiSize = uiTransform[ "_contentSize" ];
            if( uiSize == null ) continue;

            uiSize.width = size.width;
            uiSize.height = size.height;
            jsData[ "_sizeMode" ] = 2;
        }
    }

    modifyUUIDInJsonData( jsonData ){
        const self = this;
        if( jsonData == null ) return false;

        //Sprite 已处理,此处跳过
        if( self.checkSpriteType(jsonData) ){
            return false;
        }

        if( !self.checkJsonTypeHasUUID(jsonData["__type__"]) ) return false;
        
        for( const jsKey in jsonData ){
            if( jsKey == "__uuid__" ){
                const oldUUID = jsonData[ jsKey ];
                const newUUID = self._mapReplaceUUID.get( oldUUID );
                if( newUUID == null ) continue;

                jsonData[jsKey] = newUUID;
            }else if( typeof jsonData[jsKey] == "object" ){
                self.modifyUUIDInJsonData( jsonData[jsKey] );
            }
        }
    }

    modifyFile(){        
        console.log( "--------modify file--------" );
        const self = this;
        let bError = false;
        for( const fullName of self._arrModifyFile ){
            console.log( "modify file: " + fullName );
            const jsonData = FileTool.readJsonInFile( fullName );
            if( jsonData == null || !Array.isArray(jsonData) ){
                bError = true;
                console.log( "read json failt: " + fullName );
                break;
            }

            self.modifySpriteInFileJson( jsonData );
            self.modifyUUIDInJsonData( jsonData );
            FileTool.writeJsonToFile( fullName, jsonData );
        }

        return !bError;
    }

    logModifyFile(){
        console.log("\n--------modify file list--------");
        const self = this;
        self._arrModifyFile.forEach((fullName)=>{
            console.log( fullName );
        });
    }

    modifyBundleMeta( dir, bBundle ){
        const self = this;
        const name = self.getDirName( dir );
        const parentDir = self.getParentDir( dir ) + "/";
        const fileName = parentDir + name + ".meta";

        const jsonData = FileTool.readJsonInFile( fileName );
        if( bBundle ){
            jsonData[ "userData" ] = { isBundle: true };
        }else{
            jsonData[ "userData" ] = {};
        }
        FileTool.writeJsonToFile( fileName, jsonData );
        console.log( `modify dir meta file:${fileName} isBundle:${bBundle}`  );
    }

    moveBundle( srcDir, destDir ){
        //console.log( "--------move bundle--------" );
        const self = this;
        
        const srcName = self.getDirName( srcDir );
        const srcParentDir = self.getParentDir( srcDir );

        const destName = self.getDirName( destDir );
        const destParentDir = self.getParentDir( destDir );

        const newSrcDir = destParentDir + "/" + srcName + "/";
        console.log( `${srcDir} => ${newSrcDir}` );
        FileTool.rename( srcDir, newSrcDir );
        
        const oldSrcMetaName = srcParentDir + "/" + srcName + ".meta";
        const newSrcMetaName = destParentDir + "/" + srcName + ".meta";
        console.log( `${oldSrcMetaName} => ${newSrcMetaName}` );
        FileTool.rename( oldSrcMetaName, newSrcMetaName );

        const newDestDir = srcParentDir + "/" + destName + "/";
        console.log( `${destDir} => ${newDestDir}` );
        FileTool.rename( destDir, newDestDir );

        const oldDestMetaName = destParentDir + "/" + destName + ".meta";
        const newDestMetaName = srcParentDir + "/" + destName + ".meta";
        console.log( `${oldDestMetaName} => ${newDestMetaName}` );
        FileTool.rename( oldDestMetaName, newDestMetaName );
    }

    switchBundle(){
        console.log("\n--------switch bundle--------");
        const self = this;
        self.modifyBundleMeta( self._defaultDir, true );
        self.modifyBundleMeta( self._newDir, false );
        self.moveBundle( self._defaultDir, self._newDir );
    }

    changeTxt(){
        const self = this;
        const txtFileName = "txt.json";
        const tool = new I18ChangeTxtBundle( self._assetDir, self._defaultDir + txtFileName, self._newDir + txtFileName );
        const arrModify = tool.run();        
        for( const fileName of arrModify ){
            self.pushModifyFile( fileName );
        }
    }
    
    run(){        
        console.log( "===========start change bundle===========" );

        const self = this;
        self.initAsset();
        self.initLanguageDir();

        if( !self.findReplaceUUID() ){
            console.log( "please check error then try again !" );
            return;
        }
        
        if( !self.modifyFile() ){
            console.log( "please check error then try again !" );
            return;
        }

        self.changeTxt();
        self.switchBundle();
        self.logModifyFile();
        
        console.log( "===========change bundle finished===========" );
    }

}