import { _decorator, Component, Node, Input, EventTouch, UITransform, Touch, director } from 'cc';
const { ccclass, property } = _decorator;

@ccclass('JoyStick')
export class JoyStick extends Component {
    @property(Node)
    touchNode: Node = null;

    @property(Node)
    controlNode: Node = null;

    @property(Node)
    pointNode: Node = null;

    @property("")
    eventStar: string = "JoyStick_Start";
    @property("")
    eventMove: string = "JoyStick_Move";
    @property("")
    eventEnd: string = "JoyStick_End";

    private halfWidth: number = 0;
    private halfHeight: number = 0;

    private startTouch: Touch = null;

    start() {
        this.halfHeight = this.node.getComponent(UITransform).height * 0.5;
        this.halfWidth = this.node.getComponent(UITransform).width * 0.5;
        this.controlNode.active = false;

        this.touchNode.on(Input.EventType.TOUCH_START, this.onTouchStart, this);
        this.touchNode.on(Input.EventType.TOUCH_MOVE, this.onTouchMove, this);
        this.touchNode.on(Input.EventType.TOUCH_END, this.onTouchEnd, this);
        this.touchNode.on(Input.EventType.TOUCH_CANCEL, this.onTouchEnd, this);
    }

    protected onDestroy(): void {
        this.touchNode.off(Input.EventType.TOUCH_START, this.onTouchStart, this);
    }

    onTouchStart(event: EventTouch) {
        const touches = event.getTouches();
        touches.forEach(touch=>{
            if (!this.startTouch) {
                const x = touch.getUILocationX();
                const y = touch.getUILocationY();
                this.startTouch = touch;
                this.controlNode.active = true;
                this.controlNode.setPosition(x - this.halfWidth, y - this.halfHeight, 0);
                this.pointNode.setPosition(0,0,0);
                director.getScene().emit(this.eventStar);
            }
        });
    }
    onTouchMove(event: EventTouch) {
        const touches = event.getTouches();
        touches.forEach(touch=>{
            if(this.startTouch && this.startTouch.getID() == touch.getID()) {
                const x = touch.getUILocationX();
                const y = touch.getUILocationY();
                let pos = this.controlNode.position;
                let ox = x - this.halfWidth - pos.x;
                let oy = y - this.halfHeight - pos.y;
                let len = Math.sqrt(ox * ox + oy * oy);
                if (len <= 0) return;

                let dx = ox/len;
                let dy = oy/len;
                let r = this.controlNode.getComponent(UITransform).width * 0.5;
                if (len > r) {
                    len = r;
                    ox = dx * r;
                    oy = dy * r;
                }

                this.pointNode.setPosition(ox, oy, 0);

                let degree = Math.atan(dy/dx) / Math.PI * 180;
                if (dx < 0) degree += 180;
                else degree += 360;

                console.log(degree, len/r);
                director.getScene().emit(this.eventMove, degree, len/r);
            }
        })
    }
    onTouchEnd(event: EventTouch) {
        const touches = event.getTouches();
        touches.forEach(touch=>{
            if(this.startTouch && this.startTouch.getID() == touch.getID()) {
                this.startTouch = null;
                this.controlNode.active = false;
                director.getScene().emit(this.eventEnd);
            }
        });
    }
}

