const { ccclass, property } = cc._decorator;

enum ToolType { Brush = 0, Eraser = 1 }
enum BrushHeadType { Round = 0, Square = 1, Texture = 2 }

@ccclass
export default class Helloworld extends cc.Component {
    // ====== 节点引用 ======
    @property(cc.Node)
    bg: cc.Node = null;                     // 画布容器（缩放/平移都作用于它）


    @property(cc.Camera) //烘焙相机
    cam: cc.Camera = null;

    @property(cc.Graphics)
    work: cc.Graphics = null;               // 工作层（当前笔/橡皮绘制）

    @property(cc.Sprite)
    historySprite: cc.Sprite = null;        // 显示历史纹理（可不拖，代码会自动创建）

    // ====== 材质（可在检查器拖，也可代码加载） ======
    @property(cc.Material)
    brushMaterial: cc.Material = null;

    @property(cc.Material)
    eraserMaterial: cc.Material = null;

    // ====== 工具/笔头参数 ======
    @property({ type: cc.Enum(ToolType) })
    currentTool: ToolType = ToolType.Brush;

    @property({ type: cc.Enum(BrushHeadType) })
    brushHeadType: BrushHeadType = BrushHeadType.Round;

    @property({ type: [cc.Texture2D], tooltip: "贴图笔头列表（可选）" })
    brushHeadTextures: cc.Texture2D[] = [];

    @property
    brushHeadIndex: number = 0;

    @property({ tooltip: "画笔线宽" })
    brushWidth: number = 40;

    @property(cc.Color)
    brushColor: cc.Color = cc.Color.RED;

    @property({ tooltip: "橡皮线宽" })
    eraserWidth: number = 80;

    @property({ tooltip: "采样间距（像素）" })
    minSegment: number = 2;

    // ====== 缩放 ======
    @property({ tooltip: "滚轮缩放倍率（>1放大）" })
    zoomStep: number = 1.1;

    @property
    minScale: number = 0.2;

    @property
    maxScale: number = 5;

    @property({ tooltip: "是否以鼠标为缩放中心" })
    zoomAtCursor: boolean = true;

    // ====== 运行时对象 ======
    private _rt: cc.RenderTexture = null;   // 历史纹理
    // private _cam: cc.Camera = null;         // 烘焙相机（把 Work 烘进 _rt）
    private _drawing = false;
    private _lastPt: cc.Vec2 = null;

    private _workMask = 0;

    // clearFlags 常量（不清颜色 => 累计绘制）
    private _CLR_COLOR = (cc.Camera.ClearFlags as any).COLOR || 1;
    private _CLR_DEPTH = (cc.Camera.ClearFlags as any).DEPTH || 2;
    private _CLR_STENCIL = (cc.Camera.ClearFlags as any).STENCIL || 4;
    private get _NO_COLOR(): number { return this._CLR_DEPTH | this._CLR_STENCIL; }
    private get _WITH_COLOR(): number { return this._CLR_COLOR | this._CLR_DEPTH | this._CLR_STENCIL; }

    // ===================== 生命周期 =====================
    onLoad() {
        // 事件（绘制 & 滚轮）
        this.bg.on(cc.Node.EventType.TOUCH_START, this.onTouchStart, this);
        this.bg.on(cc.Node.EventType.TOUCH_MOVE, this.onTouchMove, this);
        this.bg.on(cc.Node.EventType.TOUCH_END, this.onTouchEnd, this);
        this.bg.on(cc.Node.EventType.TOUCH_CANCEL, this.onTouchEnd, this);
        this.bg.on(cc.Node.EventType.MOUSE_WHEEL, this.onMouseWheel, this);

        // 确保工作层存在
        if (!this.work) {
            const node = new cc.Node("Work");
            node.parent = this.bg;
            this.work = node.addComponent(cc.Graphics);
        }
    }

    start() {
        this.initHistoryRT();           // 初始化历史纹理 & 历史Sprite
        this.initBakeCamera();          // 初始化烘焙相机
        this.applyToolStyle();          // 设置工作层材质 & 样式
        (window as any).graphics = this.work; // 调试便捷

        this.setupWorkMask();
    }

    private setupWorkMask() {
        const groupList: string[] = (cc.game as any).groupList;
        const idx = groupList.indexOf('work');
        if (idx < 0) {
            cc.warn('[draw] 请在 Project Settings > Group 新建一个名为 "work" 的分组');
            return;
        }
        this.work.node.group = 'work';
        this._workMask = 1 << idx;
    }

    // ===================== 初始化：历史RT & Sprite =====================
    private initHistoryRT() {
        const w = Math.max(1, Math.round(this.bg.width));
        const h = Math.max(1, Math.round(this.bg.height));

        this._rt = new cc.RenderTexture();
        // 2.4.x
        this._rt.initWithSize(w, h);

        // if (!this.historySprite) {
        //     const node = new cc.Node("HistorySprite");
        //     node.parent = this.bg;
        //     this.historySprite = node.addComponent(cc.Sprite);
        //     this.historySprite.sizeMode = cc.Sprite.SizeMode.CUSTOM;
        //     node.setContentSize(w, h);
        //     node.setPosition(cc.v2(0, 0));
        // }

        // const sf = new cc.SpriteFrame();
        // (sf as any).setTexture ? (sf as any).setTexture(this._rt) : (sf as any)._setTexture(this._rt);
        // this.historySprite.spriteFrame = sf;
    }

    // ===================== 初始化：烘焙相机 =====================
    private initBakeCamera() {
        // const camNode = new cc.Node("BakeCam");
        // camNode.parent = this.bg;       // 放在 bg 下：与 bg 同坐标系/缩放
        // camNode.setPosition(cc.v2(0, 0));

        // this._cam = camNode.addComponent(cc.Camera);
        // this._cam.ortho = true;
        // this._cam.orthoSize = this.bg.height * 0.5;  // 正交相机覆盖 bg 高度
        // this._cam.backgroundColor = cc.color(0, 0, 0, 0);

        // // 第一次：清空到透明
        // this._cam.clearFlags = this._WITH_COLOR;
        // this.cam.targetTexture = this._rt;

        // // 暂时隐藏所有子节点，渲染一次“纯透明”
        // const toRestore: cc.Node[] = [];
        // this.bg.children.forEach(n => {
        //     if (n !== camNode) {
        //         if (n.active) { toRestore.push(n); n.active = false; }
        //     }
        // });
        // this._cam.render(); // 清透明
        // toRestore.forEach(n => n.active = true);

        // 之后：不再清颜色，累计绘制
        // this._cam.clearFlags = this._NO_COLOR;
    }

    // ===================== 工具/材质样式 =====================
    private applyToolStyle() {
        const g = this.work;
        if (!g) return;

        // 线帽/连接
        switch (this.brushHeadType) {
            case BrushHeadType.Round:
                g.lineCap = cc.Graphics.LineCap.ROUND; g.lineJoin = cc.Graphics.LineJoin.ROUND; break;
            case BrushHeadType.Square:
                g.lineCap = (cc.Graphics.LineCap as any).SQUARE || cc.Graphics.LineCap.BUTT;
                g.lineJoin = cc.Graphics.LineJoin.MITER; break;
            case BrushHeadType.Texture:
                g.lineCap = cc.Graphics.LineCap.ROUND; g.lineJoin = cc.Graphics.LineJoin.ROUND; break;
        }

        // 材质
        const mat = (this.currentTool === ToolType.Brush) ? this.brushMaterial : this.eraserMaterial;
        if (mat) g.setMaterial(0, mat);

        // 线宽/颜色
        if (this.currentTool === ToolType.Brush) {
            g.lineWidth = this.brushWidth;
            g.strokeColor = this.brushColor;

            const m = g.getMaterial(0);
            if (m) {
                if (m.getProperty("lineWidth", 0) !== undefined) m.setProperty("lineWidth", this.brushWidth);
                if (m.getProperty("brushColor", 0) !== undefined) {
                    const c = this.brushColor; m.setProperty("brushColor", [c.r / 255, c.g / 255, c.b / 255, c.a / 255]);
                }
                if (this.brushHeadType === BrushHeadType.Texture) {
                    const tex = this.brushHeadTextures[this.brushHeadIndex] || null;
                    if (tex && m.getProperty("brushTex", 0) !== undefined) m.setProperty("brushTex", tex);
                }
            }
        } else {
            // 橡皮：材质负责 DST_OUT 混合
            g.lineWidth = this.eraserWidth;
            g.strokeColor = cc.Color.WHITE; // 颜色无所谓，混合靠材质
            const m = g.getMaterial(0);
            if (m && m.getProperty("lineWidth", 0) !== undefined) m.setProperty("lineWidth", this.eraserWidth);
        }
    }

    // ======= 外部按钮：切换工具/笔头 =======
    public onClickBrush() { this.currentTool = ToolType.Brush; this.applyToolStyle(); }
    public onClickEraser() { this.currentTool = ToolType.Eraser; this.applyToolStyle(); }
    public onClickHeadRound() { this.brushHeadType = BrushHeadType.Round; this.applyToolStyle(); }
    public onClickHeadSquare() { this.brushHeadType = BrushHeadType.Square; this.applyToolStyle(); }
    public onClickHeadTexture(i: number = 0) {
        this.brushHeadType = BrushHeadType.Texture;
        this.brushHeadIndex = Math.max(0, Math.min(i, this.brushHeadTextures.length - 1));
        this.applyToolStyle();
    }

    // ===================== 坐标换算/绘制 =====================
    private uiToLocal(event: cc.Event.EventTouch): cc.Vec2 {
        const p = event.getLocation();
        return this.bg.convertToNodeSpaceAR(p); // Work 与 HistorySprite 都在 bg 下，坐标统一
    }

    private onTouchStart(e: cc.Event.EventTouch) {
        if (!this.work) return;

        // 每次落笔刷新样式（允许动态改宽/材质）
        this.applyToolStyle();

        const p = this.uiToLocal(e);
        this._drawing = true;
        this._lastPt = p.clone();

        this.work.moveTo(p.x, p.y);
        const r = this.work.lineWidth * 0.5;
        this.work.circle(p.x, p.y, r); // 起笔补圆避免断头
        this.work.fill();
        this.work.moveTo(p.x, p.y);
    }

    private onTouchMove(e: cc.Event.EventTouch) {
        if (!this._drawing || !this.work) return;
        const p = this.uiToLocal(e);
        if (this._lastPt && p.sub(this._lastPt).mag() < this.minSegment) return;

        this.work.lineTo(p.x, p.y);
        this.work.stroke();
        // this.work.moveTo(p.x, p.y);

        this._lastPt = p;
    }

    private onTouchEnd() {
        if (!this.work) return;
        this.work.stroke();      // 收笔兜底
        this.commitWork();       // ✅ 把 Work 烘进历史纹理
        this._drawing = false;
        this._lastPt = null;
    }



    /** 确保烘焙相机和 bg 对齐 */
    private fitBakeCamToBG() {
        if (!this.cam || !this.bg) return;

        // 位置同步
        this.cam.node.position = this.bg.position;

        // 缩放同步
        this.cam.node.scale = this.bg.scale;

        // ortho 尺寸同步（保证正交相机能完整覆盖 bg）
        this.cam.ortho = true;
        this.cam.orthoSize = (this.bg.height * 0.5) * this.bg.scaleY;
    }



    // ===================== 烘焙：把 Work 累计进历史纹理（仅渲染 work 组） =====================
    private commitWork() {
        if (!this.cam || !this._rt || !this.work) return;
        this.getPixelData();
        this.work.clear();
        return;

        // 1. 确保相机和 bg 对齐
        this.fitBakeCamToBG();
        // 记录相机原状态
        const oldTarget = this.cam.targetTexture;
        const oldMask = this.cam.cullingMask;
        const oldClear = this.cam.clearFlags;

        // 只渲染 work 组到 _rt，且不清颜色 => 叠加到历史
        this.cam.targetTexture = this._rt;
        if (this._workMask !== 0) this.cam.cullingMask = this._workMask;
        this.cam.clearFlags = this._NO_COLOR;

        // 如果你实现了“随缩放对齐相机”的 fitBakeCamToBG，建议这里确保对齐一次
        // this.fitBakeCamToBG();

        this.cam.render();

        // 还原相机
        // this.cam.targetTexture = oldTarget;
        // this.cam.cullingMask = oldMask;
        // this.cam.clearFlags = oldClear;

        // 清空工作层（clear 会重置状态，记得恢复样式）
        this.work.clear();
        this.applyToolStyle();
    }


    // 获取像素数据
    getPixelData() {
        const width = this._rt.width;
        const height = this._rt.height;
        const pixelData = new Uint8Array(width * height * 4); // RGBA 数据
        // 使用 renderTexture 获取像素数据
        this.cam.targetTexture = this._rt;
        this.cam.clearFlags = this._NO_COLOR;
        this.cam.render();
        let result = this.filpYImage(this._rt.readPixels(), this._rt.width, this._rt.height);
        // let result = this._rt.readPixels(pixelData, 0, 0, width, height)

        let rt = new cc.RenderTexture();
        rt.initWithData(result, cc.Texture2D.PixelFormat.RGBA8888, width, height);
        const sf = new cc.SpriteFrame();
        (sf as any).setTexture ? (sf as any).setTexture(rt) : (sf as any)._setTexture(rt);
        this.historySprite.spriteFrame = sf;

        // pixelData 就是你绘制的内容的像素数据，可以进行后续处理
        // console.log(result);
    }

    // This is a temporary solution
    filpYImage(data, width, height) {
        // create the data array
        let picData = new Uint8Array(width * height * 4);
        let rowBytes = width * 4;
        for (let row = 0; row < height; row++) {
            let srow = height - 1 - row;
            let start = srow * width * 4;
            let reStart = row * width * 4;
            // save the piexls data
            for (let i = 0; i < rowBytes; i++) {
                picData[reStart + i] = data[start + i];
            }
        }
        return picData;
    }

    /** 清屏：把历史RT清透明并清空工作层 */
    public clearAll() {
        if (!this.cam || !this._rt) return;

        // 清透明：临时隐藏子节点，开启清色
        const camNode = this.cam.node;
        const toRestore: cc.Node[] = [];
        this.bg.children.forEach(n => {
            if (n !== camNode) { if (n.active) { toRestore.push(n); n.active = false; } }
        });

        this.cam.targetTexture = this._rt;
        this.cam.clearFlags = this._WITH_COLOR;
        this.cam.backgroundColor = cc.color(0, 0, 0, 0);
        this.cam.render();

        // 还原显示 & 恢复累计模式
        toRestore.forEach(n => n.active = true);
        this.cam.clearFlags = this._NO_COLOR;

        // 清空工作层
        this.work.clear();
        this.applyToolStyle();
    }

    // ===================== 滚轮缩放（作用在 bg） =====================
    private onMouseWheel(e: cc.Event.EventMouse) {
        if (!this.bg) return;
        const scrollY = e.getScrollY();
        if (!scrollY) return;

        let factor = scrollY > 0 ? this.zoomStep : 1 / this.zoomStep;
        const old = this.bg.scale;
        let next = cc.misc.clampf(old * factor, this.minScale, this.maxScale);
        factor = next / old;
        if (factor === 1) return;

        if (!this.zoomAtCursor) { this.bg.scale = next; return; }

        const screen = e.getLocation();
        const localBefore = this.bg.convertToNodeSpaceAR(screen);
        this.bg.scale = next;
        const worldAfter = this.bg.convertToWorldSpaceAR(localBefore);
        const deltaWorld = screen.sub(worldAfter);

        const parent = this.bg.parent || this.node.parent || this.node;
        const p0 = parent.convertToNodeSpaceAR(worldAfter);
        const p1 = parent.convertToNodeSpaceAR(worldAfter.add(deltaWorld));
        const deltaInParent = p1.sub(p0);
        this.bg.position = this.bg.position.add(deltaInParent);
    }
}
