package com.lib.libgooglepay;

import android.app.Activity;
import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.widget.Toast;

import androidx.annotation.NonNull;

import com.android.billingclient.api.BillingClient;
import com.android.billingclient.api.BillingClientStateListener;
import com.android.billingclient.api.BillingFlowParams;
import com.android.billingclient.api.BillingResult;
import com.android.billingclient.api.ConsumeParams;
import com.android.billingclient.api.ConsumeResponseListener;
import com.android.billingclient.api.ProductDetails;
import com.android.billingclient.api.ProductDetailsResponseListener;
import com.android.billingclient.api.Purchase;
import com.android.billingclient.api.PurchasesResponseListener;
import com.android.billingclient.api.PurchasesUpdatedListener;
import com.android.billingclient.api.QueryProductDetailsParams;
import com.android.billingclient.api.QueryPurchasesParams;
import com.google.common.collect.ImmutableList;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.FormBody;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;

public class GooglePayUtils {
    private Activity _activity;
    private String mNotifyUrl = "";

    private static GooglePayUtils instance;
    private BillingClient billingClient;
    private Map<String, ProductDetails> productDetailsMap = new HashMap();

    private GooglePayUtils() {
    }

    public static GooglePayUtils getInstance() {
        if (instance == null) {
            instance = new GooglePayUtils();
        }
        return instance;
    }

    private final PurchasesUpdatedListener purchasesUpdatedListener = (billingResult, purchases) -> {
        int code = billingResult.getResponseCode();
        if (code == BillingClient.BillingResponseCode.OK && purchases != null) {
            onSuccess(purchases);
        } else if (code == BillingClient.BillingResponseCode.ITEM_ALREADY_OWNED && purchases != null) {
            onSuccess(purchases);
        } else {
            onFail(code);
        }
    };

    public void doInit(Activity activity, String notifyUrl) {
        mNotifyUrl = notifyUrl;
        _activity = activity;
        billingClient = BillingClient.newBuilder(activity)
                .setListener(purchasesUpdatedListener)
                .enablePendingPurchases()
                .build();
        InstallReferrerClientUtils.getInstance().doInit(activity);
        //链接到google play
        this.connectBillPay();

    }

    private void connectBillPay() {
        billingClient.startConnection(new BillConnectListener());
    }

    //连接到GooglePay
    public void connectGooglePay(String userId, String orderId, String skuId) {
        if (billingClient == null) {
            //TODO客户端同步回调支付失败，原因是为链接到google或者google的支付服务不能使用
            onFail(BillingClient.BillingResponseCode.SERVICE_TIMEOUT);
            return;
        }
        //查询商品详情
        checkSku(userId, orderId, skuId);
        queryHistory();
    }

    //查询商品
    private void checkSku(String userId, String orderId, String skuId) {
        if (productDetailsMap.containsKey(skuId)) {
            buyIt(userId, orderId, productDetailsMap.get(skuId));
            return;
        }
        ProductDetailsResponseListener productDetailsResponseListener = (billingResult, productDetailsList) -> {
            // check billingResult
            // process returned productDetailsList
            int code = billingResult.getResponseCode();
            if (code != BillingClient.BillingResponseCode.OK || productDetailsList.isEmpty()) {
                //Log.e(TAG, "查询商品失败    code = " + code + "    msg = " + msg);
                onFail(code);
                return;
            }
            //Log.e(TAG, "查询商品成功");
            buyIt(userId, orderId, productDetailsList.get(0));
            for (ProductDetails skuDetails : productDetailsList) {
                // 缓存
                if (!productDetailsMap.containsKey(skuDetails.getProductId())) {
                    productDetailsMap.put(skuDetails.getProductId(), skuDetails);
                }
            }

        };
        List<QueryProductDetailsParams.Product> productList = ImmutableList.of(
                QueryProductDetailsParams.Product.newBuilder()
                        .setProductId(skuId)
                        .setProductType(BillingClient.ProductType.INAPP)
                        .build());
        QueryProductDetailsParams queryProductDetailsParams =
                QueryProductDetailsParams.newBuilder()
                        .setProductList(productList)
                        .build();

        billingClient.queryProductDetailsAsync(queryProductDetailsParams, productDetailsResponseListener);
    }

    private void queryProductDetailsAsync(String skuId) {
        if (billingClient == null) {
            return;
        }
        ProductDetailsResponseListener productDetailsResponseListener = (billingResult, productDetailsList) -> {
            // check billingResult
            // process returned productDetailsList
            int code = billingResult.getResponseCode();
            if (code != BillingClient.BillingResponseCode.OK || productDetailsList.isEmpty()) {
                //Log.e(TAG, "查询商品失败    code = " + code + "    msg = " + msg);
                queryProductDetailsAsync("");
            }
        };
        List<QueryProductDetailsParams.Product> productList = ImmutableList.of(
                QueryProductDetailsParams.Product.newBuilder()
                        .setProductId(skuId)
                        .setProductType(BillingClient.ProductType.INAPP)
                        .build());
        QueryProductDetailsParams queryProductDetailsParams =
                QueryProductDetailsParams.newBuilder()
                        .setProductList(productList)
                        .build();

        billingClient.queryProductDetailsAsync(queryProductDetailsParams, productDetailsResponseListener);
    }

    //购买
    private void buyIt(String userId, String orderId, ProductDetails productDetails) {
        ImmutableList<BillingFlowParams.ProductDetailsParams> productDetailsParamsList =
                ImmutableList.of(
                        BillingFlowParams.ProductDetailsParams.newBuilder()
                                // retrieve a value for "productDetails" by calling queryProductDetailsAsync()
                                .setProductDetails(productDetails)
                                // to get an offer token, call ProductDetails.getSubscriptionOfferDetails()
                                // for a list of offers that are available to the user
                                // .setOfferToken(selectedOfferToken)
                                .build()
                );

        BillingFlowParams billingFlowParams = BillingFlowParams.newBuilder()
                .setProductDetailsParamsList(productDetailsParamsList)
                .setObfuscatedProfileId(orderId)
                .setObfuscatedAccountId(userId)
                .build();

        // Launch the billing flow
        BillingResult billingResult = billingClient.launchBillingFlow(_activity, billingFlowParams);
        int code = billingResult.getResponseCode();
        if (code != BillingClient.BillingResponseCode.OK) {
            onFail(code);
        }

    }


    private void handlePurchase(List<Purchase> list) {
        // Purchase retrieved from BillingClient#queryPurchasesAsync or your PurchasesUpdatedListener.

        // Verify the purchase.
        // Ensure entitlement was not already granted for this purchaseToken.
        // Grant entitlement to the user.
        if (list == null || list.isEmpty() || billingClient == null) {
            return;
        }
        for (Purchase purchase : list) {
            if (purchase == null || purchase.getPurchaseState() != Purchase.PurchaseState.PURCHASED)
                continue;
            consumeAsync(purchase);
        }

    }

    private void consumeAsync(Purchase purchase) {
        if (purchase == null || purchase.getPurchaseState() != Purchase.PurchaseState.PURCHASED) {
            assert purchase != null;
            onFail(purchase.getPurchaseState());
            return;
        }
        ConsumeParams consumeParams =
                ConsumeParams.newBuilder()
                        .setPurchaseToken(purchase.getPurchaseToken())
                        .build();

        ConsumeResponseListener listener = (billingResult, purchaseToken) -> {
            int code = billingResult.getResponseCode();
            if (code == BillingClient.BillingResponseCode.OK) {
                // Handle the success of the consume operation.
                notifyPayResult(purchase);
            }
        };

        billingClient.consumeAsync(consumeParams, listener);
    }

    private void onFail(int code) {
        //自己写关于支付失败后的操作
        String errorMsg = "FAIL";
        switch (code) {
            case -1:
                errorMsg = "SERVICE_DISCONNECTED";
                break;
            case -2:
                errorMsg = "FEATURE_NOT_SUPPORTED";
                break;
            case -3:
                errorMsg = "SERVICE_TIMEOUT";
                break;
            case -4:
                errorMsg = "NOTIFY SERVICE FAIL";
                break;
            case 1:
                errorMsg = "USER_CANCELED";
                break;
            case 2:
                errorMsg = "SERVICE_UNAVAILABLE";
                break;
            case 3:
                errorMsg = "BILLING_UNAVAILABLE";
                break;
            case 4:
                errorMsg = "ITEM_UNAVAILABLE";
                break;
            case 5:
                errorMsg = "DEVELOPER_ERROR";
                break;
            case 6:
                errorMsg = "ERROR";
                break;
            case 7:
                errorMsg = "ITEM_ALREADY_OWNED";
                break;
            case 8:
                errorMsg = "ITEM_NOT_OWNED";
                break;
        }
        

    }

    private void onSuccess(List<Purchase> list) {
        //自己写关于支付成功后的操作
        handlePurchase(list);
    }

    /**
     * 查询最近的购买交易
     */
    public void queryHistory() {
        if (billingClient == null) {
            return;
        }
        PurchasesResponseListener mPurchasesResponseListener = (billingResult, purchases) -> {
            // check billingResult
            // process returned purchase list, e.g. display the plans user owns
            int code = billingResult.getResponseCode();
            if (code != BillingClient.BillingResponseCode.OK) {
                onFail(code);
                return;
            }
            handlePurchase(purchases);
        };

        billingClient.queryPurchasesAsync(QueryPurchasesParams.newBuilder().setProductType(BillingClient.ProductType.INAPP).build(), mPurchasesResponseListener);

    }

    private void notifyPayResult(Purchase purchase) {
        OkHttpClient okHttpClient = new OkHttpClient();
        FormBody.Builder forBody = new FormBody.Builder();
        forBody.add("token", purchase.getPurchaseToken());
        forBody.add("productId", purchase.getProducts().get(0));
        forBody.add("packageName", purchase.getPackageName());
        forBody.add("orderId", Objects.requireNonNull(Objects.requireNonNull(purchase.getAccountIdentifiers()).getObfuscatedProfileId()));
        forBody.add("googleVer", getVersionName(_activity));
        forBody.add("versionCode", String.valueOf(getVersionCode(_activity)));
        forBody.add("uid", Objects.requireNonNull(purchase.getAccountIdentifiers().getObfuscatedAccountId()));
        RequestBody requestBody = forBody.build();
        Request request = new Request.Builder().url(mNotifyUrl).post(requestBody).build();
        Call call = okHttpClient.newCall(request);
        call.enqueue(new Callback() {
            @Override
            public void onFailure(@NonNull Call call, @NonNull IOException e) {
            }

            @Override
            public void onResponse(@NonNull Call call, @NonNull Response response) throws IOException {
                assert response.body() != null;
                String result = response.body().string();
                
            }
        });
    }

    class BillConnectListener implements BillingClientStateListener {

        @Override
        public void onBillingSetupFinished(BillingResult billingResult) {
            int code = billingResult.getResponseCode();
            if (code == BillingClient.BillingResponseCode.OK) {
                //链接到google服务
//                queryHistory();
                queryProductDetailsAsync("");
            }
        }

        @Override
        public void onBillingServiceDisconnected() {
            //未链接到google服务
            connectBillPay();
        }
    }

    private String getVersionName(Context context) {
        PackageManager manager = context.getPackageManager();
        String name;
        try {
            PackageInfo info = manager.getPackageInfo(context.getPackageName(), 0);
            name = info.versionName;
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
            return "";
        }

        return name;
    }

    private int getVersionCode(Context context) {
        PackageManager manager = context.getPackageManager();
        int code = 0;
        try {
            PackageInfo info = manager.getPackageInfo(context.getPackageName(), 0);
            code = info.versionCode;
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
        }
        return code;
    }


}
