import "../env";
import { _decorator, Component, Node, Prefab, instantiate, Camera } from 'cc';
const { ccclass, property } = _decorator;

import { GateClient } from '../../shared/gateClient/gateClient';
import { GameClient } from "../../shared/gameClient/gameClient";
import { IGameServerInfo } from "../../shared/gateClient/GameServer";
import { ServerListItem } from "../ui/ServerListItem";
import { OTWGameManager } from "./OTWGameManager";
import { OTWGameTouchController } from "./OTWGameTouchController";

@ccclass('OTWSceneManager')
export class OTWSceneManager extends Component {

    @property(Prefab)
    ServerInfoListItemTpl?: Prefab;
    @property(Node)
    ServerListWrap?: Node;

    @property(Node)
    ViewServerList?: Node;
    @property(Node)
    ViewGameUI?: Node;
    @property(Node)
    ViewGame3D?: Node;

    @property(Camera)
    public MainCamera?: Camera;

    @property(OTWGameManager)
    public GameMgr!: OTWGameManager;
    @property(OTWGameTouchController)
    public TouchCtrl!: OTWGameTouchController;

    public gateClient?: GateClient;
    public gameClient?: GameClient;

    onLoad() {
        this.GameMgr.sceneMgr = this;
        this.TouchCtrl.sceneMgr = this;
    }
    start() {
        this.gateClient = globalThis.getGateClient("http://127.0.0.1:7100");
        this.refreshList();

        this.stopGame();
    }

    async refreshList() {
        if (!this.ServerInfoListItemTpl || !this.ServerListWrap) return;
        var serverListRet = await this.gateClient!.getGameServers();
        if (!serverListRet.isSucc) {
            this.gateClient!.client.logger?.error("获取游戏服务器列表失败:" + serverListRet.err);
            return;
        }
        this.ServerListWrap.removeAllChildren();
        serverListRet.res.GameServerList.forEach(serverInfo => {
            let itemNode = instantiate(this.ServerInfoListItemTpl!);
            var siComp = itemNode.getComponent(ServerListItem);
            if (siComp) {
                siComp.serverInfo = serverInfo;
                siComp.onClick = (sv) => this.onServerItemClick(sv);
                this.ServerListWrap?.addChild(itemNode);
            }
        });
    }

    async onServerItemClick(sv: IGameServerInfo) {
        if (!this.gateClient) return;

        //从入口服务器获取登录token,这是模拟登录的实现(正常需要输入用户名或密码,或者第三方授权,在入口服务器获得登录token)
        var ret = await this.gateClient.loginByCustomUserName(sv, "testUser");
        if (!ret.succ) {
            return this.gateClient.client.logger?.error(ret.err);
        }

        //登录到游戏服务器
        this.gameClient = globalThis.getGameClient(sv.serverWSUrl);
        this.gameClient.disconnectHandler = () => {
            return true;
        }
        this.gameClient.onReconnectResult = (succ, err) => {
            if(!succ){
                this.stopGame();
            }
            return;
        };
        var err = await this.gameClient.login(ret.data!);
        if (err) {
            //登录失败, 简单抛个错误
            return this.gateClient.client.logger?.error(err);
        }

        if (this.ViewServerList) this.ViewServerList.active = false;
        if (this.ViewGameUI) this.ViewGameUI.active = true;
        if (this.ViewGame3D) this.ViewGame3D.active = true;

        this.startGame();
    }
    async onExitGameClick() {
        this.stopGame();
    }

    stopGame() {
        if (this.ViewServerList) this.ViewServerList.active = true;
        if (this.ViewGameUI) this.ViewGameUI.active = false;
        if (this.ViewGame3D) this.ViewGame3D.active = false;

        this.GameMgr!.exitGame();
    }

    startGame() {
        this.GameMgr!.enterGame();
    }

    update(dt: number) {
    }
    lateUpdate() {
    }
}