import { _decorator, Component, ProgressBar, Label, isValid, Color, ScrollView, Node, Prefab, instantiate } from 'cc';
import { GameEvent, GameEventType, UIColor } from './GameType';
const { ccclass, property } = _decorator;

@ccclass('UIController')
export class UIController extends Component {
    @property(ProgressBar)
    hpBar: ProgressBar = null!;

    @property(Label)
    statusLabel: Label = null!;

    @property(ScrollView)
    private scrollView: ScrollView = null!;

    @property(Prefab)
    private messageItemPrefab: Prefab = null!;

    private messages: Node[] = [];
    private readonly MAX_MESSAGES = 50;

    onLoad() {
        this.initializeUIComponents();
        this.setupEventListeners();
        this.showWelcomeMessage();
    }

    private initializeUIComponents() {
        this.safeInitializeComponent(this.hpBar, () => {
            this.hpBar.progress = 1;
            this.hpBar.node.active = true;
        });

        this.safeInitializeComponent(this.statusLabel, () => {
            this.statusLabel.string = "";
            this.statusLabel.node.active = true;
        });

        this.scrollView = this.node.getComponentInChildren(ScrollView);
        if (!this.scrollView) {
            console.error("ScrollView component not found!");
            return;
        }

        this.messages = [];
    }

    private safeInitializeComponent<T extends Component>(component: T | null, initAction: () => void) {
        if (isValid(component)) {
            initAction();
        } else {
            console.warn(`Component initialization failed: component is not valid`);
        }
    }

    private showGameOver() {
        this.safeInitializeComponent(this.statusLabel, () => {
            this.statusLabel.string = "游戏结束！";
            this.statusLabel.color = new Color().fromHEX(UIColor.NORMAL);
        });
    }

    private updateHPUI(event: { current: number; max: number }) {
        this.safeInitializeComponent(this.hpBar, () => {
            this.hpBar.progress = event.current / event.max;
        });
    }

    private showWelcomeMessage() {
        this.showMessage("欢迎来到迷宫探险！", UIColor.TUTORIAL);
        this.showMessage("点击控制按钮开始移动", UIColor.TUTORIAL);
        this.showMessage("剩余步数显示在右上方", UIColor.TUTORIAL);
    }

    private setupEventListeners() {
        GameEvent.on(GameEventType.HP_CHANGED, this.updateHPUI, this);
        GameEvent.on(GameEventType.CHARACTER_DEAD, this.showGameOver, this);
        GameEvent.on(GameEventType.GAME_RESET, this.resetUI, this);
        GameEvent.on(GameEventType.SYSTEM_MESSAGE, this.onSystemMessage, this);
    }

    public showMessage(content: string, color: string = UIColor.NORMAL) {
        if (!this.scrollView || !this.messageItemPrefab) return;

        const messageNode = instantiate(this.messageItemPrefab);
        messageNode.active = true; // 可能需要显式设置
        messageNode.setPosition(0, 0); // 检查坐标是否正确
        const contentLabel = messageNode.getComponentInChildren(Label);
        if (contentLabel) {
            contentLabel.string = content;
            contentLabel.color = new Color().fromHEX(color);
        }

        // 使用ScrollView的content节点作为容器
        const contentView = this.scrollView.node.getChildByName("view")?.getChildByName("content");
        if (contentView) {
            messageNode.setParent(contentView);
            this.messages.push(messageNode);
        }

        while (this.messages.length > this.MAX_MESSAGES) {
            const oldestMessage = this.messages.shift();
            if (oldestMessage) {
                oldestMessage.destroy();
            }
        }

        this.scheduleOnce(() => {
            if (this.scrollView && this.scrollView.isValid) {
                this.scrollView.scrollToBottom();
            }
        }, 0);
    }

    public resetUI() {
        this.safeInitializeComponent(this.hpBar, () => {
            this.hpBar.progress = 1;
            this.hpBar.node.active = true;
        });

        this.safeInitializeComponent(this.statusLabel, () => {
            this.statusLabel.string = "";
            this.statusLabel.color = new Color().fromHEX(UIColor.DANGER);
        });

        this.messages = [];
        
        this.showWelcomeMessage();
    }

    private onSystemMessage(event: { text: string; color?: string }) {
        this.showMessage(event.text, event.color || UIColor.NORMAL);
    }

    protected onDestroy() {
        GameEvent.off(GameEventType.HP_CHANGED, this.updateHPUI, this);
        GameEvent.off(GameEventType.CHARACTER_DEAD, this.showGameOver, this);
        GameEvent.off(GameEventType.GAME_RESET, this.resetUI, this);
        GameEvent.off(GameEventType.SYSTEM_MESSAGE, this.onSystemMessage, this);
    }
}