import { resources, assetManager, Asset } from "cc";

/**
 * 资源加载类
 * 1. 加载完成后自动记录引用关系，根据DependKeys记录反向依赖
 * 2. 支持资源使用，如某打开的UI使用了A资源，其他地方释放资源B，资源B引用了资源A，如果没有其他引用资源A的资源，会触发资源A的释放，
 * 3. 能够安全释放依赖资源（一个资源同时被多个资源引用，只有当其他资源都释放时，该资源才会被释放）
 */

// 资源加载的处理回调
export type ProcessCallback = (completedCount: number, totalCount: number, item: void) => void;
// 资源加载的完成回调
export type CompletedCallback = (error: Error, resource: void | void[]) => void;

export class LoadResArgs {
    urls: string | string[];
    type: typeof Asset;
    onCompleted?: CompletedCallback;
    onProgess?: ProcessCallback;
};

export default class ResLoader {
    /**
         * 适配资源加载参数
         * @param urls 
         * @param type 
         * @param onProgess 
         * @param onCompleted 
         */
    public static parseLoadResArgs(urls, type, onCompleted, onProgess): LoadResArgs {
        // if (onCompleted === undefined) {
        //     if (typeof type == 'function') {
        //         if (typeof onProgess == 'function') {
        //             onCompleted = onProgess;
        //             // onProgess = type;
        //         } else {
        //             onCompleted = type;
        //         }
        //     } else if (typeof onProgess == 'function') {
        //         onCompleted = onProgess;
        //         onProgess = null;
        //     }
        // }
        return { urls, type, onProgess, onCompleted };
    }
    /**
     * 开始加载资源
     * @param url           资源url
     * @param type          资源类型，默认为null
     * @param onProgess     加载进度回调
     * @param onCompleted   加载完成回调
     * @param use           资源使用key，根据makeUseKey方法生成
     */
    public loadRes(url: string | string[]): void;
    public loadRes(url: string | string[], onCompleted: CompletedCallback): void;
    public loadRes(url: string | string[], onProgess: ProcessCallback, onCompleted: CompletedCallback): void;
    public loadRes(url: string | string[], type: typeof Asset): void;
    public loadRes(url: string | string[], type: typeof Asset, onCompleted: CompletedCallback): void;
    public loadRes(url: string | string[], type: typeof Asset, onProgess: ProcessCallback, onCompleted: CompletedCallback): void;
    public loadRes() {
        resources.load.apply(resources, arguments);
    }

    public loadResDir(dir: string): void;
    public loadResDir(url: string, onCompleted: CompletedCallback): void;
    public loadResDir(url: string, onProgess: ProcessCallback, onCompleted: CompletedCallback): void;
    public loadResDir(url: string, type: typeof Asset): void;
    public loadResDir(url: string, type: typeof Asset, onCompleted: CompletedCallback): void;
    public loadResDir(url: string, type: typeof Asset, onProgess: ProcessCallback, onCompleted: CompletedCallback): void;
    public loadResDir() {
        resources.loadDir.apply(resources, arguments);
    }

    public loadRemoteRes<T extends Asset>(url: string, options: Record<string, void>, onComplete: (err: Error, asset: T) => void): void;
    public loadRemoteRes<T extends Asset>(url: string, options: Record<string, void>): void;
    public loadRemoteRes<T extends Asset>(url: string, onComplete: (err: Error, asset: T) => void): void;
    public loadRemoteRes<T extends Asset>(url: string): void;
    public loadRemoteRes() {
        assetManager.loadRemote.apply(assetManager, arguments);
    }

    public releaseArray(assets: Asset[]) {
        for (let i = 0; i < assets.length; ++i) {
            this.releaseAsset(assets[i]);
        }
    }

    /**
     * 直接通过asset释放资源（如Prefab、SpriteFrame）
     * @param asset 要释放的asset
     */
    public releaseAsset(asset: Asset) {
        asset.decRef();
    }
}

export let resLoader: ResLoader = new ResLoader();

