import { Sample } from "./Sample";
import { Mission } from "./Mission/Mission";
import { Monster } from "./Role/Monster";
import { JsonAsset, error, js, log, resources } from "cc";
import { Prop } from "./Prop/Prop";
import { Equip } from "./Prop/Equip/Equip";
import { FightMapPath } from "./FightMapPath";
import { Stage } from "./Stage";

/**
 * 数据模版管理器
 * @author pop.T
 */
export default class SampleManager {
	/**
	 * 所有数据模版集合
	 */
	private static m_allCfgDict = {};

	/**
	 * 获取某个数据的模版
	 * @param tType T类
	 * @param key 下标
	 * @example let testCfg:TestCfg = getSample<TestCfg>(TestCfg, 1001);
	 */
	public static getSample<T extends Sample>(tType: { new(): T }, key: number): T {
		let table = SampleManager.getAllSamples<T>(tType);
		if (table == null)
			return null;
		let result = table[String(key)];
		return result;
	}

	/**
	 * 读取某个数据的克隆对象
	 * @param tType T类
	 * @param key 下标
	 * @example let testCfg:TestCfg = newSample<TestCfg>(TestCfg, 1001);
	 */
	public static newSample<T extends Sample>(tType: { new(): T }, key: number): T {
		let table = SampleManager.getAllSamples<T>(tType);
		if (table == null)
			return null;
		let result = table[String(key)].clone();
		return result;
	}


	/**
	 * 读取某个表中所有数据
	 * @param tType T类
	 * @return 
	 * @example let allTestCfg = getAllSamples<TestCfg>(TestCfg);
	 */
	public static getAllSamples<T extends Sample>(tType: { new(): T }) {
		let configName: string = tType["name"];
		if (this.m_allCfgDict[configName] == null) {
			log(" has no sample: tType=" + tType + ", name=" + name)
			return null;
		}
		return this.m_allCfgDict[configName];
	}

	/**
	 * 解析配置文件中的数据
	 * @param tType T类
	 * @return 
	 * @example let allTestCfg = getAllSamples<TestCfg>(TestCfg);
	 */
	public static parseAllSamples<T extends Sample>(tType: { new(): T }) {
		let configName: string = tType["name"];
		let self = this;
		if (this.m_allCfgDict[configName] == null) {
			//加载单张表 用到再加载
			this.m_allCfgDict[configName] = {};
			resources.load("sample/" + configName, (err, res) => {
				if (err) {
					log(" load " + configName + " err, message=" + err);
				}
				let jsonAsset = res as JsonAsset;
				if (jsonAsset && jsonAsset.json) {
					for (let key in jsonAsset.json) {
						let sample;
						// 此处进行class获取的判定, 如果配置有className字段, 则用那个字段来声明对象用于同类型的子类实现(比如任务)
						let className = jsonAsset.json[key].className;
						if (!className)// 如果没有配置, 则直接用文件名来作为默认的类名
							className = configName;
						//通过className 反射获得对象
						let sampleClass = js.getClassByName(className);
						sample = new sampleClass();
						sample.init(jsonAsset.json[key]);
						self.m_allCfgDict[configName][key] = sample;
					}
					log(" load " + configName + ", success");
				}
			});
		}
		return this.m_allCfgDict[configName];
	}

}

/**
 * 将配置文件初始化
 */
export async function initAllSample() {
	log('start init all samples');
	await SampleManager.parseAllSamples<Mission>(Mission);
	// SampleManager.parseAllSamples<Monster>(Monster);
	await SampleManager.parseAllSamples<Prop>(Prop);
	await SampleManager.parseAllSamples<Equip>(Equip);
	await SampleManager.parseAllSamples<Stage>(Stage);
	await SampleManager.parseAllSamples<FightMapPath>(FightMapPath);
}