const {ccclass, property} = cc._decorator;

export const playerState = {
    stand: 1,
    jump: 2,
    attack: 3
}

@ccclass
export default class Player extends cc.Component {

    @property(cc.Float)
    speedForce : number =  200.0;

    @property(cc.Float)
    jumpForce: number = 200;

    /** 角色的移动速度 */
    private speed = cc.v2(0, 0);

    /** 角色的当前行动状态 */
    private state = playerState.stand;

    /** 角色的当前动作名称 */
    private currentAnim = 'player_idle';

    /** 角色的动画组件 */
    private playerAni: cc.Animation | null = null;

    /** 角色的操作输入 */
    private keyInput = {};

    /** 角色的连续攻击计数 */
    private combo: number = 0;

    /** 记录角色当前施展连续攻击剩余的时间 */
    private leftDelayTime: number = 0;

    public getState() {
        return this.state;
    }

    onLoad() {
        this.playerAni = this.node.getComponent(cc.Animation);
        cc.systemEvent.on('keydown', this.onKeyDown, this);
        cc.systemEvent.on('keyup', this.onKeyUp, this);
    }

    onDestroy() {
        cc.systemEvent.off('keydown', this.onKeyDown, this);
        cc.systemEvent.off('keyup', this.onKeyUp, this);

    }

    onKeyDown(event: cc.Event.EventKeyboard) {
        this.keyInput[event.keyCode] = 1;
    }

    onKeyUp(event: cc.Event.EventKeyboard) {
        this.keyInput[event.keyCode] = 0;
    }

    update(dt) {
        // 获取当前刚体的移动速度
        let linearVelocity = this.node.parent.getComponent(cc.RigidBody).linearVelocity;

        // 判断角色是否攻击
        if (this.state == playerState.stand) {
            if (this.keyInput[cc.macro.KEY.j]) {
                if (this.combo == 0) {
                    this.setAni('player_lightattack_01', playerState.attack);
                    linearVelocity.x = -10 * this.node.scaleX;
                    this.leftDelayTime = 0.6;
                    this.combo++;
                } else if (this.combo == 1) {
                    this.setAni('player_lightattack_02', playerState.attack);
                    linearVelocity.x = -10 * this.node.scaleX;
                    this.leftDelayTime = 0.6;
                    this.combo++;
                } else if (this.combo == 2) {
                    this.setAni('player_lightattack_03', playerState.attack);
                    linearVelocity.x = -15 * this.node.scaleX;
                    this.leftDelayTime = 0;
                    this.combo = 0;
                }
                this.node.parent.getComponent(cc.RigidBody).linearVelocity = linearVelocity;
            } else if (this.keyInput[cc.macro.KEY.k]) {
                if (this.combo == 0) {
                    this.setAni('player_heavyattack_01', playerState.attack);
                    linearVelocity.x = -20 * this.node.scaleX;
                    this.leftDelayTime = 1.4;
                    this.combo++;
                } else if (this.combo == 1) {
                    this.setAni('player_heavyattack_02', playerState.attack);
                    linearVelocity.x = -25 * this.node.scaleX;
                    this.leftDelayTime = 1.4;
                    this.combo++;
                } else if (this.combo == 2) {
                    this.setAni('player_heavyattack_03', playerState.attack);
                    linearVelocity.x = -35 * this.node.scaleX;
                    this.leftDelayTime = 0;
                    this.combo = 0;
                }
                this.node.parent.getComponent(cc.RigidBody).linearVelocity = linearVelocity;
            }
        }

        // 假定攻击时无法移动或者跳跃
        if (this.state != playerState.attack) {
            // 获取当前是否按下方向键
            if (this.keyInput[cc.macro.KEY.a]) {
                this.speed.x = -1;
                this.node.scaleX = 1;
                if (this.state == playerState.stand) {
                    this.setAni('player_run', playerState.stand);
                }
            } else if (this.keyInput[cc.macro.KEY.d]) {
                this.speed.x = 1;
                this.node.scaleX = -1;
                if (this.state == playerState.stand) {
                    this.setAni('player_run', playerState.stand)
                }
            } else {
                this.speed.x = 0;
                if (this.state == playerState.stand) {
                    this.setAni('player_idle', playerState.stand)
                }
            }

            // 判断当前是否处于跳跃状态
            if (this.state == playerState.stand) {
                if (this.keyInput[cc.macro.KEY.space]) {
                    linearVelocity.y = this.jumpForce;
                    this.setAni('player_jump', playerState.jump)
                }    
            }

            // 处理新的移动速度
            if (this.speed.x) {
                linearVelocity.x = this.speed.x * this.speedForce;
            } else {
                this.speed.x = 0;
            }
            this.node.parent.getComponent(cc.RigidBody).linearVelocity = linearVelocity;
        }
    
        // 如果角色处于跳跃状态
        if (this.state == playerState.jump) {
            if (linearVelocity.y < 0) {
                this.setAni('player_fall', playerState.jump);
            }
        }

        // 判断角色是否重置连续攻击计数
        if (this.leftDelayTime != 0) {
            this.leftDelayTime -= dt;
            if (this.leftDelayTime <= 0) {
                this.leftDelayTime = 0;
                this.combo = 0;
            }
        }
    }

    setAni(anim: string, state: number) {
        if (this.currentAnim == anim) return;
        this.state = state;
        this.currentAnim = anim;
        this.playerAni.play(anim);
    }

    resetAttack() {
        this.state = playerState.stand;
    }

    onBeginContact(contact, self: cc.Collider, other: cc.Collider) {
        if (other.node.group == 'ground') {
            if (self.tag == 1 && this.state == playerState.jump) {
                this.setAni('player_falldown', playerState.jump);
            }
        }
    }

    printColliderOffset() {
        const thisNodeOffset = this.node.getComponent(cc.BoxCollider).offset;
        const childNodeOffset = this.node.getChildByName('player_attack').getComponent(cc.BoxCollider).offset;
        cc.log('This node offset = ', thisNodeOffset.x.toFixed(3), thisNodeOffset.y.toFixed(3));
        cc.log('Child node offset = ', childNodeOffset.x.toFixed(3), childNodeOffset.y.toFixed(3));
        cc.log('----------------------------------------------');
    }
}
